<?php

/**
 * Attribute request for adding new application
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2014 Lifestyle Webconsulting GmbH
 * @version    $Id:$
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Model\Request\UserApplicationAttribute;

use Sso\WebserviceBundle\Entity\Webservice\Type\Attribute;
use Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationAttribute;

/**
 * Attribute request
 */
class Add extends Base
{

    /**
     * List of Sso\WebserviceBundle\Entity\Webservice\Type\Attribute
     * @var array
     */
    protected $attributes = array();

    /**
     * List of Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationAttribute
     * @var array
     */
    protected $userApplicationAttributes = array();

    /**
     * Init attribute model
     * @param \DOMDocument $simpleXML
     */
    public function init($simpleXML)
    {
        parent::init($simpleXML);
        $this->setUserModel($this->_factoryUser($simpleXML->Key->UserType->children()));
        $this->setApplicationModel($this->_factoryApplication($simpleXML->Key->ApplicationType->children()));
        foreach ($simpleXML->Value->xpath('AttributeType') as $attribute) {
            $this->addUserApplicationAttribute($this->_factoryUserApplicationAttribute($attribute->children()));
            $this->addAttribute($this->_factoryAttribute($attribute->children()));
        }
    }

    /**
     * Build user object
     * @param \DOMElement $xml
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\User
     */
    protected function _factoryUser($xml)
    {
        // Validate for application attribute set
        $validationGroups = $this->userIdentifyValdiationGroups($xml);
        array_push($validationGroups, 'UserApplicationAttribute_Set');

        return $this->typeF->user($xml, $validationGroups);
    }

    /**
     * Build attribute value object
     * @param \DOMElement $xml
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationAttribute
     */
    protected function _factoryUserApplicationAttribute($xml)
    {
        // Validate for add
        return $this->typeF->userApplicationAttribute($xml, array('UserApplicationAttribute_Set'));
    }

    /**
     * Build attribute object
     * @param \DOMElement $xml
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\Attribute
     */
    protected function _factoryAttribute($xml)
    {
        return $this->typeF->attribute($xml, array('UserApplicationAttribute_Set'));
    }

    /**
     * Build application object
     * @param \DOMElement $xml
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\Application
     */
    protected function _factoryApplication($xml)
    {
        // Validate for adding new attribute
        return $this->typeF->application($xml, array('UserApplicationAttribute_Set'));
    }

    /**
     * Set attribute value to list
     * @param \Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationAttribute $attribute
     */
    public function addUserApplicationAttribute(UserApplicationAttribute $attribute)
    {
        $this->userApplicationAttributes[] = $attribute;
    }

    /**
     * Set attribute value to list
     * @param \Sso\WebserviceBundle\Entity\Webservice\Type\Attribute $attribute
     */
    public function addAttribute(Attribute $attribute)
    {
        $key = $attribute->getLowername();
        if (!isset($this->attributes[$key])) {
            $this->attributes[$key] = $attribute;
        }
    }

    /**
     * Get list of attribute values
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationAttribute[]
     */
    public function getUserApplicationAttributes()
    {
        return $this->userApplicationAttributes;
    }

    /**
     * Get attribute from previously created attribute list
     */
    public function getAttributeByName($attributeName)
    {
        $key = strtolower($attributeName);

        return isset($this->attributes[$key]) ? $this->attributes[$key] : null;
    }

    /**
     * List of errors
     * @return array
     */
    public function getErrors()
    {
        $errors = parent::getErrors();
        foreach ($this->attributes as $attribute) {
            $errors = array_merge($errors, $attribute->errors()->getErrors());
        }
        foreach ($this->userApplicationAttributes as $attribute) {
            $errors = array_merge($errors, $attribute->errors()->getErrors());
        }

        return $errors;
    }

    /**
     * Returns a list of allowed field names in xml
     * @return string
     */
    protected function _xmlAllowedElements()
    {
        return '<Key required="true">'
        .'<UserType required="true">'
        .'<Guid />'
        .'<Email />'
        .'<Username />'
        .'<Identifier />'
        .'</UserType>'
        .'<ApplicationType required="true">'
        .'<Name required="true"></Name>'
        .'</ApplicationType>'
        .'</Key>'
        .'<Value required="true">'
        .'<AttributeType required="true" multiple="true">'
        .'<Name required="true"></Name>'
        .'<Type required="true"></Type>'
        .'<Value required="true"></Value>'
        .'</AttributeType>'
        .'</Value>';
    }
}
