<?php

/**
 * User application request
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2014 Lifestyle Webconsulting GmbH
 * @version    $Id:$
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Model\Request\UserApplicationRole;

use Sso\WebserviceBundle\Entity\Webservice\Type;

/**
 * User request add application to user
 */
class Add extends Base
{

    /**
     * List of Sso\WebserviceBundle\Entity\Webservice\Type\Role
     * @var array
     */
    protected $roles = array();

    /**
     * List of Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationRole
     * @var array
     */
    protected $userApplicationRoles = array();

    /**
     * Init attribute model
     * @param \DOMDocument $simpleXML
     */
    public function init($simpleXML)
    {
        parent::init($simpleXML);
        $this->setUserModel($this->_factoryUser($simpleXML->Key->UserType->children()));
        $this->setApplicationModel($this->_factoryApplication($simpleXML->Key->ApplicationType->children()));
        foreach ($simpleXML->Value->xpath('RoleType') as $role) {
            $this->addUserApplicationRole($this->_factoryUserApplicationRole($role->children()));
            $this->addRole($this->_factoryRole($role->children()));
        }
    }

    /**
     * Build user object
     * @param \DOMElement $xml
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\User
     */
    protected function _factoryUser($xml)
    {
        $validationGroups = $this->userIdentifyValdiationGroups($xml);
        array_push($validationGroups, 'UserApplicationRole_Add');

        return $this->typeF->user($xml, $validationGroups);
    }

    /**
     * Build application object
     * @param \DOMElement $xml
     * @return Type\Application
     */
    protected function _factoryApplication($xml)
    {
        return $this->typeF->application($xml, array('UserApplicationRole_Add'));
    }

    /**
     * Build user application role object
     * @param \DOMElement $xml
     * @return Type\UserApplicationRole
     */
    protected function _factoryUserApplicationRole($xml)
    {
        return $this->typeF->userApplicationRole($xml, array('UserApplicationRole_Add'));
    }

    /**
     * Build role object
     * @param \DOMElement $xml
     * @return Type\Role
     */
    protected function _factoryRole($xml)
    {
        // Validate for add
        return $this->typeF->role($xml, array('UserApplicationRole_Add'));
    }

    /**
     * Add role value to list
     * @param \Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationRole $role
     */
    public function addUserApplicationRole(Type\UserApplicationRole $role)
    {
        $this->userApplicationRoles[] = $role;
    }

    /**
     * Add role value to list
     * @param \Sso\WebserviceBundle\Entity\Webservice\Type\Role $role
     */
    public function addRole(Type\Role $role)
    {
        $this->roles[$role->getLowername()] = $role;
    }

    /**
     * Get list of user application roles
     * @return array
     */
    public function getUserApplicationRoles()
    {
        return $this->userApplicationRoles;
    }

    /**
     * Get role from previously created role list
     */
    public function getRoleByName($roleName)
    {
        $key = strtolower($roleName);

        return isset($this->roles[$key]) ? $this->roles[$key] : null;
    }

    /**
     * List of errors
     * @return array
     */
    public function getErrors()
    {
        $errors = parent::getErrors();
        foreach ($this->roles as $role) {
            $errors = array_merge($errors, $role->errors()->getErrors());
        }
        foreach ($this->userApplicationRoles as $role) {
            $errors = array_merge($errors, $role->errors()->getErrors());
        }

        return $errors;
    }

    /**
     * Returns a list of allowed field names in xml
     * @return string
     */
    protected function _xmlAllowedElements()
    {
        return '<Key required="true">'
        .'<UserType required="true">'
        .'<Identifier />'
        .'<Guid />'
        .'<Username />'
        .'<Email />'
        .'</UserType>'
        .'<ApplicationType required="true">'
        .'<Name required="true"></Name>'
        .'</ApplicationType>'
        .'</Key>'
        .'<Value required="true">'
        .'<RoleType required="true" multiple="true">'
        .'<Name required="true"></Name>'
        .'<Active />'
        .'</RoleType>'
        .'</Value>';
    }
}
