<?php

/**
 * UserSearch request for finding users by application role
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2015 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Model\Request\UserSearch;

use Sso\WebserviceBundle\Entity\Webservice\Type\Application as ApplicationModel;
use Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationRole as UserApplicationRoleModel;

class FindByApplicationRole extends Base
{
    /**
     * Application
     * @var ApplicationModel
     */
    protected $applicationModel;

    /**
     * Role
     * @var UserApplicationRoleModel
     */
    protected $userApplicationRole;

    /**
     * The search result: GUIDs
     * @var array
     */
    protected $userGuids;

    /**
     * @var \DateTime
     */
    protected $modifiedAfter;

    /**
     * @var boolean|null
     */
    private $userApplicationActive;

    /**
     * @var boolean|null
     */
    private $userApplicationRoleActive;

    /**
     * Init application model
     * @param \SimpleXMLElement $simpleXML
     */
    public function init($simpleXML)
    {
        parent::init($simpleXML);
        $this->setApplicationModel($this->_factoryApplication($simpleXML->Key->ApplicationType->children()));
        $this->setUserApplicationRoleModel($this->_factoryUserApplicationRole($simpleXML->Key->RoleType->children()));

        if (property_exists($simpleXML->Key->RoleType, 'Active')) {
            $this->userApplicationRoleActive = (boolean)(int)$simpleXML->Key->RoleType->Active;
        }

        if (property_exists($simpleXML->Key->ApplicationType, 'Active')) {
            $this->userApplicationActive = (boolean)(int)$simpleXML->Key->ApplicationType->Active;
        }

        if (property_exists($simpleXML->Key->UserType, 'ModifiedAfter')) {
            $this->setModifiedAfterObject((string)$simpleXML->Key->UserType->ModifiedAfter);
        }
    }

    /**
     * @return boolean|null
     */
    public function getUserApplicationActive()
    {
        return $this->userApplicationActive;
    }

    /**
     * @return boolean|null
     */
    public function getUserApplicationRoleActive()
    {
        return $this->userApplicationRoleActive;
    }

    /**
     * List of errors
     * @return array
     */
    public function getErrors()
    {
        $errors = parent::getErrors();
        if ($this->applicationModel) {
            $errors = array_merge($errors, $this->applicationModel->errors()->getErrors());
        }
        if ($this->userApplicationRole) {
            $errors = array_merge($errors, $this->userApplicationRole->errors()->getErrors());
        }

        return $errors;
    }

    /**
     * @param ApplicationModel $applicationModel
     */
    public function setApplicationModel(ApplicationModel $applicationModel)
    {
        $this->applicationModel = $applicationModel;
    }

    /**
     * @return ApplicationModel
     */
    public function getApplicationModel()
    {
        return $this->applicationModel;
    }

    /**
     * @param string $dateSting
     */
    public function setModifiedAfterObject($dateSting)
    {

        try {
            $this->modifiedAfter = new \DateTime($dateSting);
        } catch (\Exception $exc) {
            $this->modifiedAfter = null;
            $this->errors()->addError(new ExternalError('df1', 'WrongDateFormat'));
        }
    }

    /**
     * @return \DateTime|null
     */
    public function getModifiedAfter()
    {
        return $this->modifiedAfter;
    }


    /**
     * Build application object
     * @param \DOMElement $xml
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\Application
     */
    protected function _factoryApplication($xml)
    {
        // Validate for add
        return $this->typeF->application($xml, array('UserSearch_FindByRole'));
    }

    /**
     * @param UserApplicationRoleModel $userApplicationRoleModel
     */
    public function setUserApplicationRoleModel(UserApplicationRoleModel $userApplicationRoleModel)
    {
        $this->userApplicationRole = $userApplicationRoleModel;
    }

    /**
     * @return UserApplicationRoleModel
     */
    public function getUserApplicationRoleModel()
    {
        return $this->userApplicationRole;
    }

    /**
     * Build application object
     * @param \DOMElement $xml
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationRole
     */
    protected function _factoryUserApplicationRole($xml)
    {
        // Validate for search
        return $this->typeF->userApplicationRole($xml, array('UserSearch_FindByRole'));
    }

    /**
     * Set result
     * @param array $rows
     */
    public function setSearchResult($rows)
    {
        /* Make result unique */
        $searchResult = array();
        foreach ($rows as $row) {
            $searchResult[$row['Guid']] = $row;
        }
        $this->userGuids = array_values($searchResult);
    }

    /**
     * List of User-GUIDs
     * @return array
     */
    public function getSearchResult()
    {
        return $this->userGuids;
    }

    /**
     * Returns a list of allowed field names in xml
     * @return string
     */
    protected function _xmlAllowedElements()
    {
        return '<Key required="true">'
        .'<UserType>'
        .'<ModifiedAfter/>'
        .'</UserType>'
        .'<ApplicationType required="true">'
        .'<Name required="true"></Name>'
        .'<Active/>'
        .'</ApplicationType>'
        .'<RoleType required="true">'
        .'<Name required="true"></Name>'
        .'<Active/>'
        .'</RoleType>'
        .'</Key>';
    }
}