<?php

/**
 * Class UserType
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Services\User\Add\Request\Version2;

use Symfony\Component\Validator\Constraints as Assert;
use JMS\Serializer\Annotation as Serializer;

/**
 * Class UserType
 *
 * @Serializer\XmlRoot("UserType")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\WebserviceBundle\Services\User\Add\Request\Version2
 */
class UserType
{
    /**
     * @Assert\Email()
     *
     * @Serializer\SerializedName("Email")
     * @Serializer\Type("string")
     * @var string|null
     */
    protected $email;

    /**
     * @Assert\NotBlank()
     * @Assert\Regex(
     *     pattern="/\s/",
     *     match=false,
     *     message="The username contains whitespace characters which is not allowed"
     * )
     * @Assert\Length(
     *      min = 5,
     *      max = 240,
     *      minMessage = "Username must at least be {{ limit }} characters long",
     *      maxMessage = "Username can not exceed {{ limit }} characters"
     * )
     *
     * @Serializer\SerializedName("Username")
     * @Serializer\Type("string")
     * @var string
     */
    protected $username;

    /**
     * @Serializer\SerializedName("Firstname")
     * @Serializer\Type("string")
     * @var string
     */
    protected $firstname;

    /**
     * @Assert\NotBlank()
     *
     * @Serializer\SerializedName("Lastname")
     * @Serializer\Type("string")
     * @var string
     */
    protected $lastname;

    /**
     * @Assert\NotBlank()
     * @Serializer\SerializedName("Password")
     * @Serializer\Type("string")
     * @var string
     */
    protected $password;

    /**
     * @Assert\NotBlank()
     * @Assert\Regex(
     *     pattern="/(0|1)/",
     *     match=true,
     *     message="The Active Flag can only contain 0 or 1"
     * )
     * @Serializer\SerializedName("Active")
     * @Serializer\Type("integer")
     * @var integer
     */
    protected $active;

    /**
     * @Serializer\SerializedName("AuthId")
     * @Serializer\Type("string")
     * @var string
     */
    protected $authId;

    /**
     * @Serializer\SerializedName("LdapSearchAttributes")
     * @Serializer\Type("string")
     * @var string
     */
    protected $ldapSearchAttributes;

    /**
     * @Serializer\SerializedName("LdapSearchValue")
     * @Serializer\Type("string")
     * @var string
     */
    protected $ldapSearchValue;

    /**
     * @Assert\Regex(
     *     pattern="/(0|1)/",
     *     match=true,
     *     message="The MfaEnabled Flag can only contain 0 or 1"
     * )
     * @Serializer\SerializedName("MfaEnabled")
     * @Serializer\Type("integer")
     * @var integer
     */
    protected $mfaEnabled = 0;

    /**
     * @return string|null
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @return mixed
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * @return mixed
     */
    public function getFirstname()
    {
        return $this->firstname;
    }

    /**
     * @return mixed
     */
    public function getLastname()
    {
        return $this->lastname;
    }

    /**
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * @return boolean
     */
    public function isActive()
    {
        return $this->active;
    }

    /**
     * @return string
     */
    public function getAuthId()
    {
        return $this->authId;
    }

    /**
     * @return string
     */
    public function getLdapSearchAttributes()
    {
        return $this->ldapSearchAttributes;
    }

    /**
     * @return string
     */
    public function getLdapSearchValue()
    {
        return $this->ldapSearchValue;
    }

    /**
     * @return integer
     */
    public function getMfaEnabled()
    {
        return $this->mfaEnabled;
    }
}
