<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace Sso\WebserviceBundle\Services\User\Add\Request\Version4;

use Sso\WebserviceBundle\Validator\Constraints as SsoAssert;
use Symfony\Component\Validator\Constraints as Assert;
use JMS\Serializer\Annotation as Serializer;

/**
 * Class UserType
 *
 * @Serializer\XmlRoot("UserType")
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\WebserviceBundle\Services\User\Add\Request\Version4
 */
class UserType
{
    /**
     * @Assert\NotBlank()
     *
     * @Serializer\SerializedName("Email")
     * @Serializer\Type("string")
     * @var string
     */
    private $email;

    /**
     * @Assert\NotBlank()
     * @Assert\Regex(
     *     pattern="/\s/",
     *     match=false,
     *     message="The username contains whitespace characters which is not allowed"
     * )
     * @Assert\Length(
     *      min = 5,
     *      max = 240,
     *      minMessage = "Username must at least be {{ limit }} characters long",
     *      maxMessage = "Username can not exceed {{ limit }} characters"
     * )
     *
     * @Serializer\SerializedName("Username")
     * @Serializer\Type("string")
     * @var string
     */
    private $username;

    /**
     * @Serializer\SerializedName("Firstname")
     * @Serializer\Type("string")
     * @var string
     */
    private $firstname;

    /**
     * @Assert\NotBlank()
     *
     * @Serializer\SerializedName("Lastname")
     * @Serializer\Type("string")
     * @var string
     */
    private $lastname;

    /**
     * @Assert\NotBlank()
     * @Serializer\SerializedName("Password")
     * @Serializer\Type("string")
     * @var string
     */
    private $password;

    /**
     * @Assert\NotBlank()
     * @Assert\Regex(
     *     pattern="/(0|1)/",
     *     match=true,
     *     message="The Active Flag can only contain 0 or 1"
     * )
     * @Serializer\SerializedName("Active")
     * @Serializer\Type("string")
     * @var integer
     */
    private $active;

    /**
     * @Serializer\SerializedName("AuthId")
     * @Serializer\Type("string")
     * @var string
     */
    private $authId;

    /**
     * @Serializer\SerializedName("LdapSearchAttributes")
     * @Serializer\Type("string")
     * @var string
     */
    private $ldapSearchAttributes;

    /**
     * @Serializer\SerializedName("LdapSearchValue")
     * @Serializer\Type("string")
     * @var string
     */
    private $ldapSearchValue;

    /**
     * @Assert\Regex(
     *     pattern="/(0|1)/",
     *     match=true,
     *     message="The MfaEnabled Flag can only contain 0 or 1"
     * )
     * @Serializer\SerializedName("MfaEnabled")
     * @Serializer\Type("integer")
     * @var integer
     */
    private $mfaEnabled = 0;

    /**
     * @Serializer\SerializedName("PasswordPolicy")
     * @Serializer\Type("string")
     *
     * @var string
     */
    private $passwordPolicy;

    /**
     * @Assert\DateTime()
     *
     * @Serializer\SerializedName("LastPasswordChange")
     * @Serializer\Type("DateTime")
     *
     * @var \DateTime
     */
    private $lastPasswordChange;

    /**
     * @Assert\Regex(
     *     pattern="/(0|1)/",
     *     match=true,
     *     message="The PasswordExpired Flag can only contain 0 or 1"
     * )
     * @Serializer\SerializedName("PasswordExpired")
     * @Serializer\Type("string")
     *
     * @var integer
     */
    private $passwordExpired;

    /**
     * @SsoAssert\PasswordEncryptType
     * @Serializer\SerializedName("PasswordEncryptType")
     * @Serializer\Type("string")
     *
     * @var string|null
     */
    private $passwordEncryptType;

    /**
     * @Assert\Type(type="array")
     * @Serializer\SerializedName("UserGroups")
     * @Serializer\Type("array<string>")
     * @Serializer\XmlList(entry="UserGroup")
     *
     * @var array|string[]|null
     */
    private $userGroups;

    /**
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @return mixed
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * @return mixed
     */
    public function getFirstname()
    {
        return $this->firstname;
    }

    /**
     * @return mixed
     */
    public function getLastname()
    {
        return $this->lastname;
    }

    /**
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * @return boolean
     */
    public function isActive()
    {
        return $this->active;
    }

    /**
     * @return string
     */
    public function getAuthId()
    {
        return $this->authId;
    }

    /**
     * @return string
     */
    public function getLdapSearchAttributes()
    {
        return $this->ldapSearchAttributes;
    }

    /**
     * @return string
     */
    public function getLdapSearchValue()
    {
        return $this->ldapSearchValue;
    }

    /**
     * @return integer
     */
    public function getMfaEnabled()
    {
        return $this->mfaEnabled;
    }

    /**
     * @return string
     */
    public function getPasswordPolicy()
    {
        return $this->passwordPolicy;
    }

    /**
     * @return \DateTime
     */
    public function getLastPasswordChange()
    {
        return $this->lastPasswordChange;
    }

    /**
     * @return int
     */
    public function getPasswordExpired()
    {
        return $this->passwordExpired;
    }

    /**
     * @return string|null
     */
    public function getPasswordEncryptType(): ?string
    {
        return $this->passwordEncryptType;
    }

    /**
     * @return array|string[]|null
     */
    public function getUserGroups()
    {
        return $this->userGroups;
    }
}
