<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace Sso\WebserviceBundle\Services\UserApplication\ShowDetails\Request\Version3;

use JMS\Serializer\SerializerInterface;
use Sso\WebserviceBundle\ErrorHandler\ErrorHandlerInterface;
use Sso\WebserviceBundle\Services\UserApplication\ShowDetails\RequestParserInterface;
use Sso\WebserviceBundle\Services\UserApplication\ShowDetails\RequestData\Factory as RequestDataFactory;
use InvalidArgumentException;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Class RequestParser
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\Services\UserApplication\ShowDetails\Request\Version3
 */
final class RequestParser implements RequestParserInterface
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var RequestDataFactory
     */
    private $factory;

    /**
     * @var ErrorHandlerInterface
     */
    private $errorHandler;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * RequestParser constructor.
     * @param SerializerInterface $serializer
     * @param RequestDataFactory $factory
     * @param ErrorHandlerInterface $errorHandler
     * @param ValidatorInterface $validator
     */
    public function __construct(
        SerializerInterface $serializer,
        RequestDataFactory $factory,
        ErrorHandlerInterface $errorHandler,
        ValidatorInterface $validator
    ) {
        $this->serializer = $serializer;
        $this->factory = $factory;
        $this->errorHandler = $errorHandler;
        $this->validator = $validator;
    }

    /**
     * @param string $content
     * @return \Sso\WebserviceBundle\Services\UserApplication\ShowDetails\RequestData\User
     */
    public function parse($content)
    {
        /** @var Request $request */

        $request = $this->serializer->deserialize($content, Request::class, 'xml');


        if (!($request instanceof Request)) {
            throw new InvalidArgumentException();
        }

        $user = $this->factory->user();
        if (!$this->validate($request)) {
            return $user;
        }

        //fill the DTO !!!
        $userType = $request->getUserApplication()->getShowDetails()->getKey()->getUserType();
        $applicationType =  $request->getUserApplication()->getShowDetails()->getKey()->getApplicationType();

        if($applicationType){
            $applicationTypeData = $this->factory->applicationType();
            $applicationTypeData ->setNames($applicationType->getNames());
            $user->setApplicationType($applicationTypeData);
        }

        if ($identifier = $userType->getIdentifier()) {
            $user->setIdentifier($identifier);
            return $user;
        }
        if ($username = $userType->getUsername()) {
            $user->setUsername($username);
            return $user;
        }
        if ($guid = $userType->getGuid()) {
            $user->setGuid($guid);
            return $user;
        }
        if ($email = $userType->getEmail()) {
            $user->setEmail($email);
            return $user;
        }

        $this->errorHandler->addError(400, 'uasd2', 'uasd001', 'Wrong user data', 'No valid identifier found');
        return $user;
    }

    /**
     * @param Request $request
     * @return bool
     */
    private function validate(Request $request)
    {
        $validationErrors = $this->validator->validate($request);
        if (count($validationErrors) === 0) {
            return true;
        }
        foreach ($validationErrors as $error) {
            $message = $error->getPropertyPath() . ': ' . $error->getMessage();
            $this->errorHandler->addError(400, $message, $message, $message, $message);
        }
        return false;
    }
}
