<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Tests\EventListener;

use Sso\WebserviceBundle\EventListener\ApiVersionListener;
use Sso\WebserviceBundle\ServiceCommand;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpKernel\Event\GetResponseEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class ApiVersionListenerTest
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\Tests\EventListener
 */
class ApiVersionListenerTest extends \PHPUnit\Framework\TestCase
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ServiceCommand
     */
    private $serviceCommand;

    protected function setUp()
    {
        $this->logger = $this->createMock(LoggerInterface::class);
        $this->serviceCommand = new ServiceCommand();
    }

    public function testConstructorSuccess()
    {
        $object = new ApiVersionListener($this->logger, array(1, 2), $this->serviceCommand);
        $this->assertInstanceOf(ApiVersionListener::class, $object);
    }

    public function testTriggerEventSuccess()
    {
        $validApiVersions = array(1, 2);
        $event = $this->factoryGetResponseEvent(1);
        $object = new ApiVersionListener($this->logger, $validApiVersions,$this->serviceCommand);
        $object->parseHeader($event);

        $this->assertSame('Version1', $object->toString());
        $this->assertSame('Version1', (string)$object);
        $this->assertSame(1, $object->toInt());
    }

    public function testFallBackVersion()
    {
        $validApiVersions = array(1, 2);
        $event = $this->factoryGetResponseEvent(4);

        $object = new ApiVersionListener($this->logger, $validApiVersions, $this->serviceCommand);

        $object->parseHeader($event);

        $this->assertSame('Version1', $object->toString());
        $this->assertSame('Version1', (string)$object);
        $this->assertSame(1, $object->toInt());
    }

    public function testTriggerEventMissingHeader()
    {
        $validApiVersions = array(1, 2);
        $event = $this->factoryGetResponseEvent(2);
        $event->getRequest()->headers->remove('API-VERSION');

        $object = new ApiVersionListener($this->logger, $validApiVersions, $this->serviceCommand);
        $object->parseHeader($event);

        // Fallback to Version1
        $this->assertSame('Version1', $object->toString());
        $this->assertSame('Version1', (string)$object);
        $this->assertSame(1, $object->toInt());
    }

    /**
     * @param array $validApiVersions
     * @param int $apiVersion
     * @param string $apiVersionAsString
     * @param int $apiVersionAsInt
     * @dataProvider toStringSuccessDataProvider
     */
    public function testToStringSuccess($validApiVersions, $apiVersion, $apiVersionAsString, $apiVersionAsInt)
    {
        $event = $this->factoryGetResponseEvent($apiVersion);

        $object = new ApiVersionListener($this->logger, $validApiVersions, $this->serviceCommand);
        $object->parseHeader($event);

        $this->assertSame($apiVersionAsString, $object->toString());
        $this->assertSame($apiVersionAsString, (string)$object);
        $this->assertSame($apiVersionAsInt, $object->toInt());
    }

    /**
     * @return array
     */
    public function toStringSuccessDataProvider()
    {
        $data = array();
        $cntData = rand(2, 5);
        do {
            $start = rand(3, 5);
            $end = $start + rand(3, 10);
            $validApiVersions = range($start, $end);
            $apiVersion = $validApiVersions[rand(0, count($validApiVersions) - 1)];
            $data[] = array($validApiVersions, $apiVersion, 'Version' . $apiVersion, $apiVersion);
            --$cntData;
        } while (0 < $cntData);
        return $data;
    }

    /**
     * @param string|int $apiVersion
     * @return GetResponseEvent
     */
    private function factoryGetResponseEvent($apiVersion)
    {
        $request = Request::create('/', 'OPTIONS');
        $request->headers->set('API-VERSION', $apiVersion);
        $httpKernel = $this->createMock(HttpKernelInterface::class, array(), array($request));
        return new GetResponseEvent($httpKernel, $request, HttpKernelInterface::MASTER_REQUEST);
    }
}
