<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Tests\Functional\Services\Tests\Version1;

use Sso\WebserviceBundle\Tests\Functional\Services\Helper\Credentials;
use Sso\WebserviceBundle\Tests\Functional\Services\Helper\Version1\FunctionalTestCase;
use Sso\WebserviceBundle\Tests\Functional\Services\Helper\Version1\Constants;
use Sso\WebserviceBundle\Tests\Functional\Services\Helper\Version1\Model\Application;
use Sso\WebserviceBundle\Tests\Functional\Services\Helper\Version1\Model\User;

/**
 * Class UserApplicationShowDetailsTest
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Tests\Version1
 */
class UserApplicationShowDetailsTest extends FunctionalTestCase
{
    public function testWithoutApplicationAttributes()
    {
        $application = $this->helperFactory->applicationAdd()->execute();
        $this->helperFactory->userPermissionAdd()->execute($application);

        $user = $this->helperFactory->userAdd()->execute();

        $this->helperFactory->userApplicationAdd()->execute($user, $application);

        $requestBody = $this->getRequest($user);
        $response = $this->sendPostRequest('UserApplication', 'ShowDetails', $requestBody);

        static::assertEquals(200, $response->getStatusCode(), $response->getContent());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-success-without-application-attribute.xml',
            'SsoResponse'
        );
        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );
    }

    public function testWithApplicationAttributes()
    {
        $application = $this->helperFactory->applicationAdd()->execute();
        $this->helperFactory->userPermissionAdd()->execute($application);

        $applicationAttribute = $this->helperFactory->applicationAttributeAdd()->execute($application);

        $user = $this->helperFactory->userAdd()->execute();

        $this->helperFactory->userApplicationAdd()->execute($user, $application);

        $this->helperFactory->userApplicationAttributeAdd()->execute($user, $application, $applicationAttribute);

        $requestBody = $this->getRequest($user);
        $response = $this->sendPostRequest('UserApplication', 'ShowDetails', $requestBody);

        static::assertEquals(200, $response->getStatusCode());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-success-with-application-attribute.xml',
            'SsoResponse'
        );
        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );
    }

    public function testWithApplicationRoles()
    {
        $application = $this->helperFactory->applicationAdd()->execute();
        $this->helperFactory->userPermissionAdd()->execute($application);

        $applicationRole = $this->helperFactory->applicationRoleAdd()->execute($application);

        $user = $this->helperFactory->userAdd()->execute();

        $this->helperFactory->userApplicationAdd()->execute($user, $application);
        $this->helperFactory->userApplicationRoleAdd()->execute($user, $application, $applicationRole);

        $requestBody = $this->getRequest($user);
        $response = $this->sendPostRequest('UserApplication', 'ShowDetails', $requestBody);

        static::assertEquals(200, $response->getStatusCode());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-success-with-application-role.xml',
            'SsoResponse'
        );
        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );
    }

    public function testWithApplicationAttributesAndApplicationRoles()
    {
        $application = $this->helperFactory->applicationAdd()->execute();
        $this->helperFactory->userPermissionAdd()->execute($application);

        $applicationRole = $this->helperFactory->applicationRoleAdd()->execute($application);

        $applicationAttribute = $this->helperFactory->applicationAttributeAdd()->execute($application);

        $user = $this->helperFactory->userAdd()->execute();

        $this->helperFactory->userApplicationAdd()->execute($user, $application);
        $this->helperFactory->userApplicationRoleAdd()->execute($user, $application, $applicationRole);
        $this->helperFactory->userApplicationAttributeAdd()->execute($user, $application, $applicationAttribute);

        $requestBody = $this->getRequest($user);
        $response = $this->sendPostRequest('UserApplication', 'ShowDetails', $requestBody);

        static::assertEquals(200, $response->getStatusCode());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-success-with-application-attribute-and-role.xml',
            'SsoResponse'
        );

        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );
    }

    public function testInvalidUserAndValidApplication()
    {
        $faker = \Faker\Factory::create();
        $user = new User();
        $user->setEmail(Constants::PREFIX . $faker->email);
        $user->setUsername(Constants::PREFIX . $faker->userName);
        $user->setActive(true);
        $user->setFirstname($faker->firstName);
        $user->setLastname($faker->lastName);
        $user->setPassword($faker->password(8, 120) . '_Aa1');

        $application = new Application();
        $application->setName('NonExistingApplication');

        $requestBody = $this->getRequest($user);
        $response = $this->sendPostRequest('UserApplication', 'ShowDetails', $requestBody);

        static::assertEquals(200, $response->getStatusCode());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-error.xml',
            'SsoResponse'
        );
        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );

        // Value check
        $xmlSimple = simplexml_load_string($xml);
        static::assertEquals(
            Constants::STATUS_FAILURE,
            (string)$xmlSimple->Status,
            $this->fXml($requestBody, $xml, 'Global xml status code does not match!')
        );

        static::assertRegExp(
            Constants::REGEXP_DATE,
            (string)$xmlSimple->Date,
            $this->fXml($requestBody, $xml, 'Date format does not match!')
        );

        static::assertEquals(
            Constants::STATUS_FAILURE,
            (string)$xmlSimple->UserApplication->ShowDetails->Status,
            $this->fXml($requestBody, $xml, 'Xml status code does not match!')
        );

        static::assertEquals(
            'u001',
            (string)$xmlSimple->UserApplication->ShowDetails->Errors->Error->ErrorCode,
            $this->fXml($requestBody, $xml, 'Error code does not match!')
        );
    }

    /**
     * @param User $user
     * @return string
     */
    private function getRequest(User $user)
    {
        return '<?xml version="1.0" encoding="UTF-8"?>
<SsoRequest>' . (string)(new Credentials($this->configuration)) . '
    <UserApplication>
        <ShowDetails>
            <Key>
                <UserType>
                    <Username>' . $this->cdata($user->getUsername()) . '</Username>
                </UserType>
            </Key>
        </ShowDetails>
    </UserApplication>
</SsoRequest>';
    }
}
