<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Tests\Functional\Services\Tests\Version2;

use Sso\WebserviceBundle\Tests\Functional\Services\Helper\Credentials;
use Sso\WebserviceBundle\Tests\Functional\Services\Helper\Version2\FunctionalTestCase;
use Sso\WebserviceBundle\Tests\Functional\Services\Helper\Version2\Constants;
use Sso\WebserviceBundle\Tests\Functional\Services\Helper\Version2\Model\User;

/**
 * Class UserDeleteTest
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Tests\Version2
 */
class UserDeleteTest extends FunctionalTestCase
{
    public function testDeleteUserByGuid()
    {
        $user = $this->helperFactory->userAdd()->execute(false);

        $requestBody = $this->getRequestByGuid($user->getGuid());
        $response = $this->sendPostRequest('User', 'Delete', $requestBody);

        static::assertEquals(200, $response->getStatusCode());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-success.xml',
            'SsoResponse'
        );

        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );
    }

    public function testDeleteUserByUsername()
    {
        $user = $this->helperFactory->userAdd()->execute(false);

        $requestBody = $this->getRequestByUsername($user->getUsername());
        $response = $this->sendPostRequest('User', 'Delete', $requestBody);

        static::assertEquals(200, $response->getStatusCode());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-success.xml',
            'SsoResponse'
        );

        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );
    }

    public function testDeleteUserByEmail()
    {
        $user = $this->helperFactory->userAdd()->execute(false);

        $requestBody = $this->getRequestByEmail($user->getEmail());
        $response = $this->sendPostRequest('User', 'Delete', $requestBody);

        static::assertEquals(200, $response->getStatusCode());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-success.xml',
            'SsoResponse'
        );

        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );
    }

    public function testDeleteUserByIdentifier()
    {
        $user = $this->helperFactory->userAdd()->execute(false);

        $requestBody = $this->getRequestByIdentifier($user->getUsername());
        $response = $this->sendPostRequest('User', 'Delete', $requestBody);

        static::assertEquals(200, $response->getStatusCode());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-success.xml',
            'SsoResponse'
        );

        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );
    }

    public function testDeleteUnknownUser()
    {
        $faker = \Faker\Factory::create();
        $user = new User();
        $user->setEmail(Constants::PREFIX . $faker->email);
        $user->setUsername(Constants::PREFIX . $faker->userName);
        $user->setActive(false);
        $user->setFirstname($faker->firstName);
        $user->setLastname($faker->lastName);
        $user->setPassword($faker->password(8, 120) . '_Aa1');

        $requestBody = $this->getRequestByIdentifier($user->getUsername());
        $response = $this->sendPostRequest('User', 'Delete', $requestBody);

        static::assertEquals(200, $response->getStatusCode());

        $xml = $response->getContent();
        $xmlElement = $this->loadResponse($xml);

        $xmlElementExpected = $this->loadXmlDomElement(
            __DIR__ . '/xml/response-structure-error.xml',
            'SsoResponse'
        );

        static::assertEqualXMLStructure(
            $xmlElementExpected,
            $xmlElement,
            false,
            $this->fXml($requestBody, $xml, 'Xml structure check failed')
        );

        // Value check
        $xmlSimple = simplexml_load_string($xml);

        static::assertEquals(
            Constants::STATUS_FAILURE,
            (string)$xmlSimple->Status,
            $this->fXml($requestBody, $xml, 'Global xml status code does not match!')
        );

        static::assertRegExp(
            Constants::REGEXP_DATE,
            (string)$xmlSimple->Date,
            $this->fXml($requestBody, $xml, 'Date format does not match!')
        );

        static::assertEquals(
            Constants::STATUS_FAILURE,
            (string)$xmlSimple->User->Delete->Status,
            $this->fXml($requestBody, $xml, 'Xml status code does not match!')
        );

        static::assertEquals(
            'u001',
            (string)$xmlSimple->User->Delete->Errors->Error->ErrorCode,
            $this->fXml($requestBody, $xml, 'Error code does not match!')
        );
    }

    /**
     * @param string $username
     * @return string
     */
    private function getRequestByUsername($username)
    {
        return '<?xml version="1.0" encoding="UTF-8"?>
<SsoRequest>' . (string)(new Credentials($this->configuration)) . '<User>
        <Delete>
            <Key>
                <UserType>
                    <Username>' . $this->cdata($username) . '</Username>
                </UserType>
            </Key>
        </Delete>
    </User>
</SsoRequest>';
    }

    /**
     * @param string $email
     * @return string
     */
    private function getRequestByEmail($email)
    {
        return '<?xml version="1.0" encoding="UTF-8"?>
<SsoRequest>' . (string)(new Credentials($this->configuration)) . '<User>
        <Delete>
            <Key>
                <UserType>
                    <Email>' . $this->cdata($email) . '</Email>
                </UserType>
            </Key>
        </Delete>
    </User>
</SsoRequest>';
    }

    /**
     * @param string $guid
     * @return string
     */
    private function getRequestByGuid($guid)
    {
        return '<?xml version="1.0" encoding="UTF-8"?>
<SsoRequest>' . (string)(new Credentials($this->configuration)) . '<User>
        <Delete>
            <Key>
                <UserType>
                    <Guid>' . $this->cdata($guid) . '</Guid>
                </UserType>
            </Key>
        </Delete>
    </User>
</SsoRequest>';
    }

    /**
     * @param string $identifier
     * @return string
     */
    private function getRequestByIdentifier($identifier)
    {
        return '<?xml version="1.0" encoding="UTF-8"?>
<SsoRequest>' . (string)(new Credentials($this->configuration)) . '<User>
        <Delete>
            <Key>
                <UserType>
                    <Identifier>' . $this->cdata($identifier) . '</Identifier>
                </UserType>
            </Key>
        </Delete>
    </User>
</SsoRequest>';
    }
}
