<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace Sso\AccessBundle\Authorization;

use Doctrine\ORM\EntityManagerInterface;
use Sso\AccessBundle\Entity\Factory;
use Sso\AccessBundle\Entity\TemporaryToken;
use Sso\AccessBundle\Repository\TemporaryTokenRepository;
use Sso\WebserviceBundle\Security\Authentication\Token\WsFirewallToken;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;

/**
 * Class TemporaryTokenAuthorization
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package Sso\AccessBundle\Authorization
 */
class TemporaryTokenAuthorization implements TemporaryTokenAuthorizationInterface
{
    /**
     * @var EntityManagerInterface
     */
    private $objectManager;

    /**
     * @var TokenStorageInterface
     */
    private $tokenStorageInterface;

    /**
     * @var Factory
     */
    private $tokenFactory;

    /**
     * TemporaryTokenAuthorization constructor.
     * @param EntityManagerInterface $objectManager
     * @param TokenStorageInterface $tokenStorageInterface
     * @param Factory $tokenFactory
     */
    public function __construct(
        EntityManagerInterface $objectManager,
        TokenStorageInterface $tokenStorageInterface,
        Factory $tokenFactory
    ) {
        $this->objectManager = $objectManager;
        $this->tokenStorageInterface = $tokenStorageInterface;
        $this->tokenFactory = $tokenFactory;
    }

    /**
     * @return array
     */
    public function getAuthorizedUserIds(): array
    {
        $temporaryToken = $this->getTemporaryToken();
        if (null !== $temporaryToken) {
            return $temporaryToken->getUserIds();
        }

        return [];
    }

    /**
     * @param string $userId
     */
    public function addAuthorizedUserId(string $userId): void
    {
        $temporaryToken = $this->getTemporaryToken();
        if (null !== $temporaryToken) {
            $temporaryToken->addUserId($userId);
            $this->objectManager->persist($temporaryToken);
            $this->objectManager->flush($temporaryToken);
        }
    }

    /**
     * @return TemporaryToken|null
     */
    private function getTemporaryToken(): ?TemporaryToken
    {
        $token = $this->tokenStorageInterface->getToken();
        if (
            !$token instanceof WsFirewallToken ||
            !$token->accessRestricted ||
            0 === strlen($token->applicationUserIdentifier)
        ) {
            return null;
        }

        $objectRepository = $this->objectManager->getRepository(TemporaryToken::class);
        $temporaryToken = $objectRepository->find($token->applicationUserIdentifier);
        if ($temporaryToken instanceof TemporaryToken) {
            return $temporaryToken;
        }

        return $this->tokenFactory->temporaryToken($token->applicationUserIdentifier);
    }
}
