<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Api\Controller;

use Sso\WebserviceBundle\Model;
use Sso\WebserviceBundle\Api\Exception\Type\Base as BaseException;
use Sso\WebserviceBundle\Api\Exception\Type\Api as ApiException;
use Sso\WebserviceBundle\Api\Error\Type\External as ExternalError;
use Sso\WebserviceBundle\Api\Error\Type\Internal as InternalError;
use Sso\WebserviceBundle\Api;
use Sso\WebserviceBundle\Api\Response\Builder as XmlResponse;

/**
 * Class AbstractController
 * @package Sso\WebserviceBundle\Api\Controller
 */
abstract class AbstractController
{
    /**
     * @var \Sso\WebserviceBundle\Database\Webservice\Manager
     */
    protected $dbM;

    /**
     * @var Model\Factory
     */
    protected $modelFactory;

    /**
     * Element of request xml
     * 
     * @var \DOMElement
     */
    protected $xmlRequest;

    /**
     * Response xml document
     * 
     * @var XmlResponse
     */
    private $xmlResponse;

    /**
     * @var Api\Error\Storage 
     */
    public $apiErrors;

    /**
     * @var Api\Manager 
     */
    public $apiM;

    /**
     * Constructor
     * 
     * @param Api\Manager $apiM
     * @param \DOMElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    public function __construct(Api\Manager $apiM, $xmlRequest, XmlResponse $xmlResponse)
    {
        $this->modelFactory = new Model\Factory($apiM);
        $this->apiM = $apiM;
        $this->apiErrors = new Api\Error\Storage();
        $this->xmlRequest = $xmlRequest;
        $this->xmlResponse = $xmlResponse;
        $this->dbM = $apiM->database()->webservice();
    }

    /**
     * Validate controller-action against current action-list
     * 
     * @param string $actionName
     * @return bool
     */
    public function isValidControllerAction($actionName)
    {
        return in_array($actionName, $this->_getActions());
    }

    /**
     * Call controller action
     * 
     * @param string $actionName
     */
    public function callAction($actionName)
    {

        // Check controller action
        if (!$this->isValidControllerAction($actionName)) {
            $this->_exception(new ExternalError('c11', 'ActionInvalid'));
        }

        // Check xml data
        if (!property_exists($this->xmlRequest, $actionName)) {
            $this->_exception(new ExternalError('c12', 'ActionNotFound'));
        }

        // Log usage
        ApiException::setAction($actionName);
        $this->apiM->database()->serviceProvider()->tokenUsage()->setAction($actionName);

        // Check usage limit

        // Call action
        $controllerAction = lcfirst($actionName);
        foreach ($this->xmlRequest->$actionName as $request) {
            $this->xmlResponse->createAction($actionName);
            try {
                call_user_func(array($this, $controllerAction), $request, $this->xmlResponse);
            } catch (BaseException $exception) {
                $this->xmlResponse->exception($exception);
            }
        }
    }

    /**
     * Exit with error
     * 
     * @param string $actionName
     * @param Api\Error\Type\Base $error
     * @throws ApiException
     */
    private function _exception(Api\Error\Type\Base $error = null)
    {
        if (null !== $error) {
            $this->apiErrors->addError($error);
        }
        $exception = new ApiException();
        $exception->setErrors($this->apiErrors->getErrors());
        throw $exception;
    }

    /**
     * Returns list of available actions
     * 
     * @return array
     */
    abstract protected function _getActions();

    /**
     * Name of current controller
     * 
     * @return string
     */
    protected function _getControllerName()
    {
        return preg_replace('/^.*_([^_]+)$/', '$1', get_class($this));
    }
 }
