<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Api\Controller;

use Sso\WebserviceBundle\Api\Error\Type\External as ExternalError;
use Sso\WebserviceBundle\Api\Response\Builder as XmlResponse;

/**
 * Class ApplicationAttributeController
 * @package Sso\WebserviceBundle\Api\Controller
 */
class ApplicationAttributeController extends AbstractController
{
    /**
     * Returns list of available actions
     *
     * @return array
     */
    protected function _getActions()
    {
        return array(
            'Add',
            'Delete',
            'Show',
        );
    }

    /**
     * Show application attribute
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function show(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->applicationAttribute()->show($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();
        $attributeModel = $requestModel->getAttributeModel();

        //check read on application
        if(!$requestModel->hasErrors() && !in_array($requestModel->getApplicationModel()->getName(),$this->apiM->userRolesApplicationRead())){
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied', 'You are not allowed to see this Application'));
        }

        if (!$requestModel->hasErrors()) {
            if (!($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
                $requestModel->addError(new ExternalError('at201', 'ApplicationNotFound'));
            } elseif (!($attribute = $this->dbM->attribute()->getAttributeByName($application, $attributeModel->getName()))) {
                $requestModel->addError(new ExternalError('at202', 'ApplicationAttributeNotFound'));
            } else {
                $requestModel->setAttributeModel($attribute);
            }
        }

        $this->modelFactory->response($xmlResponse)->applicationAttribute()->show($requestModel);
    }

    /**
     * Add application attribute
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function add(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->applicationAttribute()->add($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();
        $attributeModel = $requestModel->getAttributeModel();

        //check write on application
        if(!$requestModel->hasErrors() && !in_array($requestModel->getApplicationModel()->getName(),$this->apiM->userRolesApplicationWrite())){
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied', 'You are not allowed to edit this Application'));
        }

        if (!$requestModel->hasErrors() && !($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
            $requestModel->addError(new ExternalError('at001', 'ApplicationNotFound'));
        }
        if (!$requestModel->hasErrors() && $this->dbM->attribute()->getAttributeByName($application, $attributeModel->getName())) {
            $requestModel->addError(new ExternalError('at002', 'AttributeExists'));
        }
        if (!$requestModel->hasErrors()) {
            $attributeModel->setApplication($application);
            $this->dbM->attribute()->saveAttribute($attributeModel);
        }
        $this->modelFactory->response($xmlResponse)->applicationAttribute()->add($requestModel);
    }

    /**
     * Delete application attribute
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function delete(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->applicationAttribute()->delete($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();
        $attributeModel = $requestModel->getAttributeModel();

        //check write on application
        if(!$requestModel->hasErrors() && !in_array($requestModel->getApplicationModel()->getName(),$this->apiM->userRolesApplicationWrite())){
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied', 'You are not allowed to edit this Application'));
        }

        if (!$requestModel->hasErrors() && !($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
            $requestModel->addError(new ExternalError('at101', 'ApplicationNotFound'));
        }
        if (!$requestModel->hasErrors() && !($attribute = $this->dbM->attribute()->getAttributeByName($application, $attributeModel->getName()))) {
            $requestModel->addError(new ExternalError('at102', 'AttributeNotFound'));
        }
        if (!$requestModel->hasErrors() && ($this->dbM->application()->getAttributeUsage($attribute) > 0)) {
            $requestModel->addError(new ExternalError('at103', 'AttributeInUse'));
        }
        if (!$requestModel->hasErrors()) {
            $this->dbM->attribute()->deleteAttribute($attribute);
        }
        $this->modelFactory->response($xmlResponse)->applicationAttribute()->delete($requestModel);
    }
}
