<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Api\Controller;

use Sso\WebserviceBundle\Api\Error\Type\External as ExternalError;
use Sso\WebserviceBundle\Api\Response\Builder as XmlResponse;

/**
 * Class ApplicationController
 * @package Sso\WebserviceBundle\Api\Controller
 */
class ApplicationController extends AbstractController
{

    /**
     * Returns list of available actions
     *
     * @return array
     */
    protected function _getActions()
    {
        return array(
            'Add',
            'Delete',
            'Show',
        );
    }

    /**
     * Show application with attributes
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function show(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->application()->show($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();

        if (!$requestModel->hasErrors() && !($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
            $requestModel->addError(new ExternalError('a011', 'ApplicationNotFound'));
        }
        //check read access on application
        if (!$requestModel->hasErrors() && !in_array($requestModel->getApplicationModel()->getName(),
                $this->apiM->userRolesApplicationRead())
        ) {
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied',
                'You are not allowed to see this Application'));
        }
        if (!$requestModel->hasErrors()) {
            $requestModel->setApplicationModel($application);
        }

        $this->modelFactory->response($xmlResponse)->application()->show($requestModel);
    }

    /**
     * Add application
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function add(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->application()->add($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();

        if (!$requestModel->hasErrors()) {
            $this->dbM->application()->saveApplication($applicationModel);
        }
        $this->modelFactory->response($xmlResponse)->application()->add($requestModel);
    }

    /**
     * Delete application
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function delete(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->application()->delete($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();

        // Search for application
        if (!$requestModel->hasErrors() && (!($application = $this->dbM->application()->getApplicationByName($applicationModel->getName())))) {
            $requestModel->addError(new ExternalError('a021', 'ApplicationNotFound'));
        }
        //check write access on application
        if (!$requestModel->hasErrors() && !in_array($requestModel->getApplicationModel()->getName(),
                $this->apiM->userRolesApplicationWrite())
        ) {
            $requestModel->addError(new ExternalError('ac002', 'ApplicationAccessDenied',
                'You are not allowed to delete this Application'));
        }

        if (!$requestModel->hasErrors() && !$this->dbM->application()->deleteApplication($application)) {
            $requestModel->addErrors($application->errors()->getErrors());
        }
        $this->modelFactory->response($xmlResponse)->application()->delete($requestModel);
    }
}
