<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Api\Controller;

use Sso\WebserviceBundle\Api\Error\Type\External as ExternalError;
use Sso\WebserviceBundle\Api\Response\Builder as XmlResponse;

/**
 * Class ApplicationRoleController
 * @package Sso\WebserviceBundle\Api\Controller
 */
class ApplicationRoleController extends AbstractController
{

    /**
     * Returns list of available actions
     *
     * @return array
     */
    protected function _getActions()
    {
        return array(
            'Add',
            'Delete',
            'Show',
        );
    }

    /**
     * Show application role
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function show(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->applicationRole()->show($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();
        $roleModel = $requestModel->getRoleModel();

        //check read on application
        if (!$requestModel->hasErrors() && !in_array($requestModel->getApplicationModel()->getName(),
                $this->apiM->userRolesApplicationRead())
        ) {
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied',
                'You are not allowed to see this Application'));
        }

        if (!$requestModel->hasErrors()) {
            if (!($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
                $requestModel->addError(new ExternalError('r031', 'ApplicationNotFound'));
            } elseif (!($role = $this->dbM->role()->getRoleByName($application, $roleModel->getName()))) {
                $requestModel->addError(new ExternalError('r032', 'ApplicationRoleNotFound'));
            } else {
                $requestModel->setRoleModel($role);
            }
        }

        $this->modelFactory->response($xmlResponse)->applicationRole()->show($requestModel);
    }

    /**
     * Add role
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function add(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->applicationRole()->add($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();
        $roleModel = $requestModel->getRoleModel();

        //check write on application
        if (!$requestModel->hasErrors() && !in_array($requestModel->getApplicationModel()->getName(),
                $this->apiM->userRolesApplicationWrite())
        ) {
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied',
                'You are not allowed to edit this Application'));
        }

        if (!$requestModel->hasErrors()) {
            if (!($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
                $requestModel->addError(new ExternalError('r011', 'ApplicationNotFound'));
            } elseif ($this->dbM->role()->getRoleByName($application, $roleModel->getName())) {
                $requestModel->addError(new ExternalError('r012', 'RoleExists'));
            }
        }
        if (!$requestModel->hasErrors()) {
            $roleModel->setApplication($application);
            $this->dbM->role()->saveRole($roleModel);
        }
        $this->modelFactory->response($xmlResponse)->applicationRole()->add($requestModel);
    }

    /**
     * Delete role
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function delete(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->applicationRole()->delete($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();
        $roleModel = $requestModel->getRoleModel();

        //check write on application
        if (!$requestModel->hasErrors() && !in_array($requestModel->getApplicationModel()->getName(),
                $this->apiM->userRolesApplicationWrite())
        ) {
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied',
                'You are not allowed to edit this Application'));
        }

        // Search for application
        if (!$requestModel->hasErrors() && !($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
            $requestModel->addError(new ExternalError('r021', 'ApplicationNotFound'));
        }

        // Search for role
        if (!$requestModel->hasErrors() && !($role = $this->dbM->role()->getRoleByName($application,
                $roleModel->getName()))
        ) {
            $requestModel->addError(new ExternalError('r022', 'RoleNotFound'));
        }

        // Delete role
        if (!$requestModel->hasErrors() && !$this->dbM->role()->deleteRole($role)) {
            $requestModel->addErrors($role->errors()->getErrors());
        }
        $this->modelFactory->response($xmlResponse)->applicationRole()->delete($requestModel);
    }
}
