<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Api\Controller;

use Sso\WebserviceBundle\Api\Error\Type\External as ExternalError;
use Sso\WebserviceBundle\Api\Response\Builder as XmlResponse;
use Sso\WebserviceBundle\Entity\Webservice\Type\UserApplication;

/**
 * Class UserApplicationController
 * @package Sso\WebserviceBundle\Api\Controller
 */
class UserApplicationController extends AbstractController
{

    /**
     * @return array
     */
    protected function _getActions()
    {
        return ['Add', 'Delete', 'Show', 'ShowDetails'];
    }

    /**
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function show(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->userApplication()->show($xmlRequest);
        $userModel = $requestModel->getUserModel();

        if (!$requestModel->hasErrors() && !($user = $this->dbM->user()->getUserByUser($userModel))) {
            $requestModel->addError(new ExternalError('ua021', 'UserNotFound'));
        }
        if (!$requestModel->hasErrors() && !$this->apiM->userAuthorization()->canUpdate($user)) {
            $requestModel->addError(new ExternalError('ua022', 'UserNotAuthorized'));
        }
        if (!$requestModel->hasErrors()) {
            $requestModel->setUserModel($user);
        }
        $this->modelFactory->response($xmlResponse)->userApplication()->show($requestModel,
            $this->apiM->userRolesApplicationRead());
    }

    /**
     * Add application to user
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function add(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->userApplication()->add($xmlRequest);
        $userModel = $requestModel->getUserModel();
        $applicationModel = $requestModel->getApplicationModel();

        if (!$requestModel->hasErrors() && !($user = $this->dbM->user()->getUserByUser($userModel))) {
            $requestModel->addError(new ExternalError('ua011', 'UserNotFound'));
        }
        if (!$requestModel->hasErrors() && !$this->apiM->userAuthorization()->canUpdate($user)) {
            $requestModel->addError(new ExternalError('ua012', 'UserNotAuthorized'));
        }
        if (!$requestModel->hasErrors() && !($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
            $requestModel->addError(new ExternalError('ua012', 'ApplicationNotFound'));
        }

        //check write on application
        if (!$requestModel->hasErrors() && !in_array($applicationModel->getName(),
                $this->apiM->userRolesApplicationWrite())
        ) {
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied',
                'You are not allowed to write this Application'));
        }

        if (!$requestModel->hasErrors() && $this->dbM->userApplication()->getUserApplication($user, $application)) {
            $requestModel->addError(new ExternalError('ua013', 'UserApplicationExists'));
        }
        if (!$requestModel->hasErrors()) {
            $userApplicationModel = new UserApplication($this->apiM->validator());
            $userApplicationModel->generateId();
            $userApplicationModel->setApplication($application);
            $userApplicationModel->setUser($user);
            $userApplicationModel->setActive($requestModel->getUserApplicationModel()->getActive());
            $this->dbM->userApplication()->saveUserApplication($userApplicationModel);
        }
        $this->modelFactory->response($xmlResponse)->userApplication()->add($requestModel);
    }

    /**
     * Delete application from user
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function delete(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->userApplication()->delete($xmlRequest);
        $userModel = $requestModel->getUserModel();
        $applicationModel = $requestModel->getApplicationModel();

        if (!$requestModel->hasErrors() && !($user = $this->dbM->user()->getUserByUser($userModel))) {
            $requestModel->addError(new ExternalError('ua031', 'UserNotFound'));
        }
        if (!$requestModel->hasErrors() && !$this->apiM->userAuthorization()->canUpdate($user)) {
            $requestModel->addError(new ExternalError('ua032', 'UserNotAuthorized'));
        }
        if (!$requestModel->hasErrors() && !($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
            $requestModel->addError(new ExternalError('ua032', 'ApplicationNotFound'));
        }
        //check write on application
        if (!$requestModel->hasErrors() && !in_array($applicationModel->getName(),
                $this->apiM->userRolesApplicationWrite())
        ) {
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied',
                'You are not allowed to write this Application'));
        }
        if (!$requestModel->hasErrors() && !($userApplication = $this->dbM->userApplication()->getUserApplication($user,
                $application))
        ) {
            $requestModel->addError(new ExternalError('ua033', 'UserApplicationNotFound'));
        }
        if (!$requestModel->hasErrors() && !$this->dbM->userApplication()->deleteUserApplication($userApplication)) {
            $requestModel->addErrors($userApplication->errors()->getErrors());
        }
        $this->modelFactory->response($xmlResponse)->userApplication()->delete($requestModel);
    }

    /**
     * Show users application list
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function showDetails(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->userApplication()->showDetails($xmlRequest);
        $userModel = $requestModel->getUserModel();

        if (!$requestModel->hasErrors() && !($user = $this->dbM->user()->getUserByUser($userModel))) {
            $requestModel->addError(new ExternalError('ua021', 'UserNotFound'));
        }
        if (!$requestModel->hasErrors()) {
            $requestModel->setUserModel($user);
        }
        $this->modelFactory->response($xmlResponse)->userApplication()->showDetails($requestModel,
            $this->apiM->userRolesApplicationRead());
    }
}
