<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Api\Controller;

use Sso\WebserviceBundle\Api\Error\Type\External as ExternalError;
use Sso\WebserviceBundle\Api\Response\Builder as XmlResponse;

/**
 * Class UserSearchController
 * @package Sso\WebserviceBundle\Api\Controller
 */
class UserSearchController extends AbstractController
{
    /**
     * Returns list of available actions
     *
     * @return array
     */
    protected function _getActions()
    {
        return array(
            'FindByApplication',
            'FindByApplicationAttribute',
            'FindByApplicationRole',
        );
    }

    /**
     * Show applications
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function findByApplication(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->userSearch()->findByApplication($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();

        if (!$requestModel->hasErrors() && !($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
            $requestModel->addError(new ExternalError('uss021', 'ApplicationNotFound'));
        }
        //check read on application
        if (!$requestModel->hasErrors() && !in_array($applicationModel->getName(),
                $this->apiM->userRolesApplicationRead())
        ) {
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied',
                'You are not allowed to read this Application'));
        }
        if (!$requestModel->hasErrors()) {
            $applicationActive = $requestModel->getUserApplicationActive();
            $modifiedAfter = $requestModel->getModifiedAfter();
            $requestModel->setSearchResult($this->dbM->user()->getGuidsByApplication($application, $applicationActive,
                $modifiedAfter));
        }

        $this->modelFactory->response($xmlResponse)->userSearch()->findByApplication($requestModel);
    }

    /**
     * Show application with attributes
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function findByApplicationAttribute(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->userSearch()->findByApplicationAttribute($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();
        $attributeModel = $requestModel->getUserApplicationAttributeModel();

        if (!$requestModel->hasErrors() && !($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
            $requestModel->addError(new ExternalError('uss011', 'ApplicationNotFound'));
        }
        //check read on application
        if (!$requestModel->hasErrors() && !in_array($applicationModel->getName(),
                $this->apiM->userRolesApplicationRead())
        ) {
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied',
                'You are not allowed to read this Application'));
        }
        if (!$requestModel->hasErrors() && !($application->hasAttribute($attributeModel))) {
            $requestModel->addError(new ExternalError('uss012', 'UserApplicationAttributeNotFound'));
        }
        if (!$requestModel->hasErrors()) {
            $modifiedAfter = $requestModel->getModifiedAfter();
            $applicationActive = $requestModel->getUserApplicationActive();
            $attributeModel->setAttribute($application->getAttribute($attributeModel->getName()));
            $requestModel->setSearchResult($this->dbM->user()->getGuidsByApplicationAttribute($attributeModel,
                $modifiedAfter, $applicationActive));
        }

        $this->modelFactory->response($xmlResponse)->userSearch()->findByApplicationAttribute($requestModel);
    }

    /**
     * Find users by application role
     *
     * @param \SimpleXMLElement $xmlRequest
     * @param XmlResponse $xmlResponse
     */
    protected function findByApplicationRole(\SimpleXMLElement $xmlRequest, XmlResponse $xmlResponse)
    {
        $requestModel = $this->modelFactory->request()->userSearch()->findByApplicationRole($xmlRequest);
        $applicationModel = $requestModel->getApplicationModel();
        $roleModel = $requestModel->getUserApplicationRoleModel();

        if (!$requestModel->hasErrors() && !($application = $this->dbM->application()->getApplicationByName($applicationModel->getName()))) {
            $requestModel->addError(new ExternalError('uss021', 'ApplicationNotFound'));
        }
        //check read on application
        if (!$requestModel->hasErrors() && !in_array($applicationModel->getName(),
                $this->apiM->userRolesApplicationRead())
        ) {
            $requestModel->addError(new ExternalError('ac001', 'ApplicationAccessDenied',
                'You are not allowed to read this Application'));
        }
        if (!$requestModel->hasErrors() && !($application->hasRole($roleModel))) {
            $requestModel->addError(new ExternalError('uss022', 'ApplicationRoleNotFound'));
        }
        if (!$requestModel->hasErrors()) {
            $modifiedAfter = $requestModel->getModifiedAfter();
            $roleModel->setRole($application->getRole($roleModel->getName()));
            $roleActive = $requestModel->getUserApplicationRoleActive();
            $applicationActive = $requestModel->getUserApplicationActive();
            $requestModel->setSearchResult($this->dbM->user()->getGuidsByApplicationRole($roleModel, $modifiedAfter,
                $roleActive, $applicationActive));
        }

        $this->modelFactory->response($xmlResponse)->userSearch()->findByApplicationRole($requestModel);
    }

}