<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Database\Webservice;

use Sso\WebserviceBundle\Entity\Webservice\Type\Attribute as ModelAttribute;
use Sso\WebserviceBundle\Entity\Webservice\Type\Application as ModelApplication;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityRepository;

/**
 * Class Application
 * @package Sso\WebserviceBundle\Database\Webservice
 */
class Application extends Base
{
    /**
     * Save application to database
     * @param ModelApplication $typeApplication
     * @return bool Returns true on success, false on validation failure
     */
    public function saveApplication(ModelApplication $typeApplication)
    {
        // Do default validation
        $typeApplication->setValidator($this->validator);
        if (!$typeApplication->validate()) {
            return false;
        }

        if ($this->readonly) {
            return false;
        }

        $this->entityManager->persist($typeApplication);
        $this->entityManager->flush($typeApplication);

        return true;
    }

    /**
     * Delete application from database
     * @param ModelApplication $typeApplication
     * @return bool True on success
     */
    public function deleteApplication(ModelApplication $typeApplication)
    {
        $typeApplication->setValidator($this->validator);
        if (!$typeApplication->validate(array('Application_Delete'))) {
            return false;
        }

        if ($this->readonly) {
            return false;
        }

        $this->entityManager->remove($typeApplication);
        $this->entityManager->flush($typeApplication);

        return true;
    }

    /**
     * Get all applications
     * @return ModelApplication[]
     */
    public function getAllApplications()
    {
        return $this->entityManager->getRepository('Sso\WebserviceBundle\Entity\Webservice\Type\Application')->findAll();
    }

    /**
     * Get application by id
     * @param string $applicationId
     * @return ModelApplication|null
     */
    public function getApplicationById($applicationId)
    {
        return $this->entityManager->getRepository('Sso\WebserviceBundle\Entity\Webservice\Type\Application')->findOneBy(array('Id' => $applicationId));
    }

    /**
     * Get application by name
     *
     * @param string $applicationName
     * @return ModelApplication|null
     */
    public function getApplicationByName($applicationName)
    {
        return $this->entityManager->getRepository('Sso\WebserviceBundle\Entity\Webservice\Type\Application')->findOneBy(array('Lowername' => strtolower($applicationName)));
    }

    /**
     * Get attribute usage
     *
     * @param ModelAttribute $applicationName
     * @return integer
     */
    public function getAttributeUsage(ModelAttribute $attribute)
    {
        return count($this->entityManager->getRepository('Sso\WebserviceBundle\Entity\Webservice\Type\UserApplicationAttribute')->findBy(array('Attribute' => $attribute)));
    }

    /**
     * Find a list of applications by application-name
     * @param string $applicationName
     * @return ModelApplication[]|null
     * @throws \Exception
     */
    public function getApplicationsByName($applicationName, $limit = 1, $offset = 0)
    {
        $applicationName = str_replace('%', '\%', strtolower($applicationName));

        return $this->getApplicationRepository()
            ->createQueryBuilder('a')
            ->where('a.Lowername LIKE :applicationname')
            ->setParameter('applicationname', $applicationName.'%')
            ->orderBy('a.Lowername')
            ->setMaxResults($limit)
            ->setFirstResult($offset)
            ->getQuery()
            ->getResult();
    }

    /**
     * @return EntityRepository
     */
    public function getApplicationRepository()
    {
        return $this->entityManager->getRepository('Sso\WebserviceBundle\Entity\Webservice\Type\Application');
    }
}
