<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Database\Webservice;

use Sso\WebserviceBundle\Entity\Webservice\Type;

/**
 * Class UserApplication
 * @package Sso\WebserviceBundle\Database\Webservice
 */
class UserApplication extends Base
{
    /**
     * Update application name in joined table
     * @param Type\Application $application
     */
    public function updateApplication(Type\Application $application)
    {
        $dql = 'UPDATE Sso\WebserviceBundle\Entity\Webservice\Type\UserApplication a '
            . 'SET a.Name=:name '
            . 'WHERE a.Application=:application';
        $query = $this->entityManager->createQuery($dql);
        $query->setParameter('name', $application->getName());
        $query->setParameter('application', $application);
        $query->execute();
    }

    /**
     * Save user-application-link to database
     * @param \Sso\WebserviceBundle\Entity\Webservice\Type\UserApplication $entity
     * @return boolean
     */
    public function saveUserApplication(Type\UserApplication $entity)
    {
        // Do default validation
        $entity->setValidator($this->validator);
        if (!$entity->isValid()) {
            return false;
        }

        if ($this->readonly) {
            return false;
        }

        $this->entityManager->persist($entity);
        $this->entityManager->flush($entity);

        //to also update the user updateDate
        $this->entityManager->flush($entity->getUser());

        return true;
    }

    /**
     * Delete user-application-link
     * @param Type\UserApplication $entity
     * @return bool True on success
     */
    public function deleteUserApplication(Type\UserApplication $entity)
    {
        $entity->setValidator($this->validator);
        if (!$entity->validate(array('UserApplication_Delete'))) {
            return false;
        }

        if ($this->readonly) {
            return false;
        }

        $this->entityManager->remove($entity);
        $this->entityManager->flush($entity);

        //to also update the user updateDate
        $this->entityManager->flush($entity->getUser());

        return true;
    }

    /**
     * Find user-application-link by userApplicationId
     * @param string $userApplicationId
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\UserApplication|null
     */
    public function getUserApplicationById($userApplicationId)
    {
        return $this->getRepository()->find($userApplicationId);
    }

    /**
     * Find user-application-link by user and application
     * @param \Sso\WebserviceBundle\Entity\Webservice\Type\User $user
     * @param \Sso\WebserviceBundle\Entity\Webservice\Type\Application $application
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\UserApplication|null
     */
    public function getUserApplication(Type\User $user, Type\Application $application)
    {
        return $this->getRepository()->findOneBy(array('User' => $user, 'Application' => $application));
    }

    /**
     * @return \Sso\WebserviceBundle\Entity\Webservice\Type\UserApplication
     */
    public function getRepository()
    {
        return $this->entityManager->getRepository('Sso\WebserviceBundle\Entity\Webservice\Type\UserApplication');
    }

    /**
     * @param Type\UserApplication $application
     */
    public function removeApplicationForce(Type\UserApplication $application)
    {
        foreach ($application->getAttributes() as $deleteAttribute) {
            $this->entityManager->remove($deleteAttribute);
        }

        foreach ($application->getRoles() as $deleteRole) {
            $this->entityManager->remove($deleteRole);
        }
        $this->entityManager->remove($application);
        $this->entityManager->flush();
    }

    /**
     * Delete user-application-link
     * @param Type\UserApplication $entity
     * @return bool True on success
     */
    public function removeUserApplication(Type\UserApplication $entity)
    {
        $entity->setValidator($this->validator);
        if (!$entity->validate(array('UserApplication_Delete'))) {
            return false;
        }

        if ($this->readonly) {
            return false;
        }

        $this->entityManager->remove($entity);
        $this->entityManager->flush();

        return true;
    }
}
