<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\PasswordCrypt;

use Sso\WebserviceBundle\Entity\Webservice\Type\User;
use Sso\WebserviceBundle\Exception\InvalidConfigurationException;

/**
 * Class PasswordCryptRepository
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\PasswordCrypt
 */
class PasswordCryptRepository implements PasswordCryptRepositoryInterface
{
    /**
     * @var string
     */
    private $defaultPasswordEncryptType;

    /**
     * @var PasswordCryptInterface[]
     */
    private $passwordCrypter = [];

    /**
     * PasswordCryptRepository constructor.
     * @param string $defaultPasswordEncryptType
     */
    public function __construct(string $defaultPasswordEncryptType)
    {
        $this->defaultPasswordEncryptType = $defaultPasswordEncryptType;
    }

    /**
     * @param PasswordCryptInterface $passwordCrypt
     */
    public function addPasswordCrypter(PasswordCryptInterface $passwordCrypt): void
    {
        $this->passwordCrypter[$passwordCrypt->getEncryptType()] = $passwordCrypt;
    }

    /**
     * @param User $user
     * @return PasswordCryptInterface
     * @throws InvalidConfigurationException
     */
    public function getByUser(User $user): PasswordCryptInterface
    {
        return $this->getByEncryptType($user->getPasswordEncryptType());
    }

    /**
     * @param string $passwordEncryptType
     * @return PasswordCryptInterface
     * @throws InvalidConfigurationException
     */
    public function getByEncryptType(string $passwordEncryptType): PasswordCryptInterface
    {
        if (!isset($this->passwordCrypter[$passwordEncryptType])) {
            throw new InvalidConfigurationException(sprintf(
                'Invalid configuration: password crypt type "%s" not found! ' .
                'Please check your configuration sso_webservice -> password_encrypt_default_type. ' .
                'Available password crypt types are: %s',
                $passwordEncryptType,
                join(', ', array_keys($this->passwordCrypter))
            ));
        }

        return $this->passwordCrypter[$passwordEncryptType];
    }

    /**
     * @return PasswordCryptInterface
     * @throws InvalidConfigurationException
     */
    public function getDefault(): PasswordCryptInterface
    {
        return $this->getByEncryptType($this->defaultPasswordEncryptType);
    }

    /**
     * Returns a list of available encryption types
     * @return string[]
     */
    public function getEncryptionTypes(): array
    {
        return array_keys($this->passwordCrypter);
    }
}
