<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Services\PasswordPolicy\Show\Request\Version3;

use JMS\Serializer\SerializerInterface;
use Sso\WebserviceBundle\ErrorHandler\ErrorHandlerInterface;
use Sso\WebserviceBundle\Services\PasswordPolicy\Show\RequestParserInterface;
use Sso\WebserviceBundle\Services\PasswordPolicy\Show\RequestData\Factory as RequestDataFactory;
use InvalidArgumentException;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Sso\WebserviceBundle\Api\PasswordPolicy\Policy\Service as PasswordPolicyService;

/**
 * Class RequestParser
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\WebserviceBundle\Services\PasswordPolicy\Show\Request\Version3
 */
final class RequestParser implements RequestParserInterface
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var RequestDataFactory
     */
    private $factory;

    /**
     * @var ErrorHandlerInterface
     */
    private $errorHandler;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * @var PasswordPolicyService
     */
    private $passwordPolicyService;

    /**
     * RequestParser constructor.
     * @param SerializerInterface $serializer
     * @param RequestDataFactory $factory
     * @param ErrorHandlerInterface $errorHandler
     * @param ValidatorInterface $validator
     */
    public function __construct(
        SerializerInterface $serializer,
        RequestDataFactory $factory,
        ErrorHandlerInterface $errorHandler,
        ValidatorInterface $validator,
        PasswordPolicyService $passwordPolicyService
    ) {
        $this->serializer = $serializer;
        $this->factory = $factory;
        $this->errorHandler = $errorHandler;
        $this->validator = $validator;
        $this->passwordPolicyService = $passwordPolicyService;
    }

    /**
     * @param string $content
     * @return \Sso\WebserviceBundle\Services\PasswordPolicy\Show\RequestData\PasswordPolicy
     */
    public function parse($content)
    {
        /** @var Request $request */
        $request = $this->serializer->deserialize($content, Request::class, 'xml');

        if (!($request instanceof Request)) {
            throw new InvalidArgumentException();
        }

        $passwordPolicy= $this->factory->passwordPolicy();
        if (!$this->validate($request)) {
            return $passwordPolicy;
        }

        if($userType = $request->getPasswordPolicy()->getShow()->getKey()->getUserType() && $passwordPolicyType = $request->getPasswordPolicy()->getShow()->getKey()->getPasswordPolicyType()){
            $this->errorHandler->addError(400, 'TooManyArguments', 'pp001', 'too many arguments', 'you can only add identifier or policy to this request not both');
            return $passwordPolicy;
        }

        if($userType = $request->getPasswordPolicy()->getShow()->getKey()->getUserType()){
            if ($identifier = $userType->getIdentifier()) {
                $passwordPolicy->setIdentifier($identifier);
                return $passwordPolicy;
            }
        }

        if($passwordPolicyType = $request->getPasswordPolicy()->getShow()->getKey()->getPasswordPolicyType()){
            if ($policyId = $passwordPolicyType->getPolicyId()) {
                $passwordPolicy->setPolicyId($policyId);
                return $passwordPolicy;
            }
        }

        //fallback to default policyId
        $defaultPolicy = $this->passwordPolicyService->getDefaultPolicy();
        if(null !== $defaultPolicy){
            $passwordPolicy->setPolicyId($defaultPolicy->getPolicyId());
            return $passwordPolicy;
        }

        $this->errorHandler->addError(400, 'pp2', 'pp002', 'Wrong data', 'No valid identifier or policyId found');
        return $passwordPolicy;
    }

    /**
     * @param Request $request
     * @return bool
     */
    private function validate(Request $request)
    {
        $validationErrors = $this->validator->validate($request);
        if (count($validationErrors) === 0) {
            return true;
        }
        foreach ($validationErrors as $error) {
            $message = $error->getPropertyPath() . ': ' . $error->getMessage();
            $this->errorHandler->addError(400, $message, $message, $message, $message);
        }
        return false;
    }
}
