<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Services\PasswordPolicy\Show\ResponseData;

use Doctrine\Common\Collections\ArrayCollection;
use Sso\WebserviceBundle\Entity\Webservice\Type\User as UserModel;
use Sso\WebserviceBundle\Entity\Webservice\Type\PasswordPolicy as PasswordPolicyModel;

/**
 * Class PasswordPolicy
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\WebserviceBundle\Services\PasswordPolicy\Show\ResponseData
 */
final class PasswordPolicy
{
    /**
     * @var string
     */
    private $guid;

    /**
     * @var string
     */
    private $email;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $firstname;

    /**
     * @var string
     */
    private $lastname;

    /**
     * @var string
     */
    private $policyId;

    /**
     * @var string
     */
    private $title;

    /**
     * @var string
     */
    private $regEx;

    /**
     * @var array
     */
    private $regexErrorMessages = [];

    /**
     * @var integer
     */
    private $validityDuration;

    /**
     * @var array
     */
    private $validityDurationErrorMessages = [];

    /**
     * @var integer
     */
    private $historySize;

    /**
     * @var array
     */
    private $historySizeErrorMessages = [];

    /**
     * @var integer
     */
    private $editDistance;

    /**
     * @var array
     */
    private $editDistanceErrorMessages = [];

    /**
     * @var integer
     */
    private $changeDelay;

    /**
     * @var array
     */
    private $changeDelayErrorMessages = [];

    /**
     * @var string
     */
    private $policyDescription;

    /**
     * @var array
     */
    private $errorMessages = [];

    /**
     * @var bool
     */
    private $isDefault;

    /**
     * @var PasswordPolicyValidations[]|ArrayCollection
     */
    private $passwordPolicyValidations;

    /**
     * @var array
     */
    private $requestMapperUser = [
        'getGuid' => 'setGuid',
        'getEmail' => 'setEmail',
        'getUsername' => 'setUsername',
        'getFirstname' => 'setFirstname',
        'getLastname' => 'setLastname',
    ];

    /**
     * @var array
     */
    private $requestMapperPolicy = [
        'getPolicyId' => 'setPolicyId',
        'getTitle' => 'setTitle',
        'getRegEx' => 'setRegEx',
        'getRegexErrorMessages' => 'setRegexErrorMessages',
        'getValidityDuration' => 'setValidityDuration',
        'getValidityDurationErrorMessages' => 'setValidityDurationErrorMessages',
        'getHistorySize' => 'setHistorySize',
        'getHistorySizeErrorMessages' => 'setHistorySizeErrorMessages',
        'getEditDistance' => 'setEditDistance',
        'getEditDistanceErrorMessages' => 'setEditDistanceErrorMessages',
        'getChangeDelay' => 'setChangeDelay',
        'getChangeDelayErrorMessages' => 'setChangeDelayErrorMessages',
        'getPolicyDescription' => 'setPolicyDescription',
        'getErrorMessages' => 'setErrorMessages',
        'isDefault' => 'setIsDefault',
    ];

    /**
     * PasswordPolicy constructor.
     */
    public function __construct()
    {
        $this->passwordPolicyValidations = new ArrayCollection();
    }

    /**
     * @param UserModel $userModel
     * @param PasswordPolicyModel $passwordPolicyModel
     */
    public function setBoth(UserModel $userModel, PasswordPolicyModel $passwordPolicyModel){
        $this->setFromUserModel($userModel);
        $this->setFromPasswordPolicyModel($passwordPolicyModel);
        return $this;
    }

    /**
     * @param UserModel $userModel
     * @return PasswordPolicy
     */
    public function setFromUserModel(UserModel $userModel)
    {
        foreach ($this->requestMapperUser as $getter => $setter) {
            if (method_exists($userModel, $getter)) {
                $this->$setter($userModel->$getter());
            }
        }
        return $this;
    }

    /**
     * @param PasswordPolicyModel $passwordPolicyModel
     * @return PasswordPolicy
     */
    public function setFromPasswordPolicyModel(PasswordPolicyModel $passwordPolicyModel)
    {
        foreach ($this->requestMapperPolicy as $getter => $setter) {
            if (method_exists($passwordPolicyModel, $getter)) {
                $this->$setter($passwordPolicyModel->$getter());
            }
        }

        //additional set validations
        foreach ($passwordPolicyModel->getPasswordPolicyValidations() as $validation) {
            $validationMapped = new PasswordPolicyValidations();
            $validationMapped->setOrderId($validation->getOrderId());
            $validationMapped->setRegEx($validation->getRegEx());
            $validationMapped->setErrorMessages($validation->getErrorMessages());
            $validationMapped->setHintMessages($validation->getHintMessages());
            $this->addPasswordPolicyValidations($validationMapped);
        }

        return $this;
    }

    /**
     * @return string
     */
    public function getGuid()
    {
        return $this->guid;
    }

    /**
     * @param string $guid
     * @return PasswordPolicy
     */
    public function setGuid($guid)
    {
        $this->guid = $guid;
        return $this;
    }

    /**
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @param string $email
     * @return PasswordPolicy
     */
    public function setEmail($email)
    {
        $this->email = $email;
        return $this;
    }

    /**
     * @return string
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * @param string $username
     * @return PasswordPolicy
     */
    public function setUsername($username)
    {
        $this->username = $username;
        return $this;
    }

    /**
     * @return string
     */
    public function getFirstname()
    {
        return $this->firstname;
    }

    /**
     * @param string $firstname
     * @return PasswordPolicy
     */
    public function setFirstname($firstname)
    {
        $this->firstname = $firstname;
        return $this;
    }

    /**
     * @return string
     */
    public function getLastname()
    {
        return $this->lastname;
    }

    /**
     * @param string $lastname
     * @return PasswordPolicy
     */
    public function setLastname($lastname)
    {
        $this->lastname = $lastname;
        return $this;
    }

    /**
     * @return string
     */
    public function getPolicyId()
    {
        return $this->policyId;
    }

    /**
     * @param string $policyId
     * @return PasswordPolicy
     */
    public function setPolicyId($policyId)
    {
        $this->policyId = $policyId;
        return $this;
    }

    /**
     * @return string
     */
    public function getTitle()
    {
        return $this->title;
    }

    /**
     * @param string $title
     * @return PasswordPolicy
     */
    public function setTitle($title)
    {
        $this->title = $title;
        return $this;
    }

    /**
     * @return string
     */
    public function getRegEx()
    {
        return $this->regEx;
    }

    /**
     * @param string $regEx
     * @return PasswordPolicy
     */
    public function setRegEx($regEx)
    {
        $this->regEx = $regEx;
        return $this;
    }

    /**
     * @return array
     */
    public function getRegexErrorMessages()
    {
        return $this->regexErrorMessages;
    }

    /**
     * @param array $regexErrorMessages
     * @return PasswordPolicy
     */
    public function setRegexErrorMessages($regexErrorMessages)
    {
        $this->regexErrorMessages = $regexErrorMessages;
        return $this;
    }

    /**
     * @return int
     */
    public function getValidityDuration()
    {
        return $this->validityDuration;
    }

    /**
     * @param int $validityDuration
     * @return PasswordPolicy
     */
    public function setValidityDuration($validityDuration)
    {
        $this->validityDuration = $validityDuration;
        return $this;
    }

    /**
     * @return array
     */
    public function getValidityDurationErrorMessages()
    {
        return $this->validityDurationErrorMessages;
    }

    /**
     * @param array $validityDurationErrorMessages
     * @return PasswordPolicy
     */
    public function setValidityDurationErrorMessages($validityDurationErrorMessages)
    {
        $this->validityDurationErrorMessages = $validityDurationErrorMessages;
        return $this;
    }

    /**
     * @return int
     */
    public function getHistorySize()
    {
        return $this->historySize;
    }

    /**
     * @param int $historySize
     * @return PasswordPolicy
     */
    public function setHistorySize($historySize)
    {
        $this->historySize = $historySize;
        return $this;
    }

    /**
     * @return array
     */
    public function getHistorySizeErrorMessages()
    {
        return $this->historySizeErrorMessages;
    }

    /**
     * @param array $historySizeErrorMessages
     * @return PasswordPolicy
     */
    public function setHistorySizeErrorMessages($historySizeErrorMessages)
    {
        $this->historySizeErrorMessages = $historySizeErrorMessages;
        return $this;
    }

    /**
     * @return int
     */
    public function getEditDistance()
    {
        return $this->editDistance;
    }

    /**
     * @param int $editDistance
     * @return PasswordPolicy
     */
    public function setEditDistance($editDistance)
    {
        $this->editDistance = $editDistance;
        return $this;
    }

    /**
     * @return array
     */
    public function getEditDistanceErrorMessages()
    {
        return $this->editDistanceErrorMessages;
    }

    /**
     * @param array $editDistanceErrorMessages
     * @return PasswordPolicy
     */
    public function setEditDistanceErrorMessages($editDistanceErrorMessages)
    {
        $this->editDistanceErrorMessages = $editDistanceErrorMessages;
        return $this;
    }

    /**
     * @return int
     */
    public function getChangeDelay()
    {
        return $this->changeDelay;
    }

    /**
     * @param int $changeDelay
     * @return PasswordPolicy
     */
    public function setChangeDelay($changeDelay)
    {
        $this->changeDelay = $changeDelay;
        return $this;
    }

    /**
     * @return array
     */
    public function getChangeDelayErrorMessages()
    {
        return $this->changeDelayErrorMessages;
    }

    /**
     * @param array $changeDelayErrorMessages
     * @return PasswordPolicy
     */
    public function setChangeDelayErrorMessages($changeDelayErrorMessages)
    {
        $this->changeDelayErrorMessages = $changeDelayErrorMessages;
        return $this;
    }

    /**
     * @return string
     */
    public function getPolicyDescription()
    {
        return $this->policyDescription;
    }

    /**
     * @param string $policyDescription
     * @return PasswordPolicy
     */
    public function setPolicyDescription($policyDescription)
    {
        $this->policyDescription = $policyDescription;
        return $this;
    }

    /**
     * @return array
     */
    public function getErrorMessages()
    {
        return $this->errorMessages;
    }

    /**
     * @param array $errorMessages
     * @return PasswordPolicy
     */
    public function setErrorMessages($errorMessages)
    {
        $this->errorMessages = $errorMessages;
        return $this;
    }

    /**
     * @return bool
     */
    public function isDefault()
    {
        return $this->isDefault;
    }

    /**
     * @param bool $isDefault
     * @return PasswordPolicy
     */
    public function setIsDefault($isDefault)
    {
        $this->isDefault = $isDefault;
        return $this;
    }

    /**
     * @return PasswordPolicyValidations[]|ArrayCollection
     */
    public function getPasswordPolicyValidations()
    {
        return $this->passwordPolicyValidations;
    }

    /**
     * @param PasswordPolicyValidations $passwordPolicyValidation
     * @return PasswordPolicy
     */
    public function addPasswordPolicyValidations(PasswordPolicyValidations $passwordPolicyValidation)
    {
        if (!$this->passwordPolicyValidations->contains($passwordPolicyValidation)) {
            $this->passwordPolicyValidations->add($passwordPolicyValidation);
        }

        return $this;
    }
}
