<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Services\User\Show;

use Sso\WebserviceBundle\ErrorHandler\ErrorHandlerInterface;
use Sso\WebserviceBundle\Event\Factory as EventFactory;
use Sso\WebserviceBundle\Event\UserAuthorizationEvent;
use Sso\WebserviceBundle\Security\Authorisation\UserAuthorization;
use Sso\WebserviceBundle\Services\HandlerInterface;
use Sso\WebserviceBundle\Services\RequestInterface as ServiceRequestInterface;
use Sso\WebserviceBundle\Services\ResponseBuilderInterface as ServiceResponseBuilderInterface;
use Sso\WebserviceBundle\Database\Webservice\User as UserRepository;
use Sso\WebserviceBundle\Entity\Webservice\Type\User as ModelUser;
use Sso\WebserviceBundle\Services\User\Show\RequestData\User as RequestDataUser;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Class Handler
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\Services\User\Show
 */
final class Handler implements HandlerInterface
{
    /**
     * @var Factory
     */
    private $factory;

    /**
     * @var UserRepository
     */
    private $database;

    /**
     * @var ErrorHandlerInterface
     */
    private $errorHandler;

    /**
     * @var UserAuthorization
     */
    private $authorization;

    /**
     * Handler constructor.
     * @param Factory $factory
     * @param UserRepository $database
     * @param ErrorHandlerInterface $errorHandler
     * @param UserAuthorization $authorization
     */
    public function __construct(
        Factory $factory,
        UserRepository $database,
        ErrorHandlerInterface $errorHandler,
        UserAuthorization $authorization
    ) {
        $this->factory = $factory;
        $this->database = $database;
        $this->errorHandler = $errorHandler;
        $this->authorization = $authorization;
    }

    /**
     * @param ServiceRequestInterface $request
     * @return ServiceResponseBuilderInterface
     */
    public function handle(ServiceRequestInterface $request)
    {
        $requestModel = $this->factory->request()->requestParser()->parse($request->getRequestBody());

        $responseBuilder = $this->factory->response()->responseBuilder();
        if ($this->errorHandler->hasErrors()) {
            return $responseBuilder;
        }

        $userModel = $this->getUserModel($requestModel);
        if (null === $userModel) {
            $this->errorHandler->addError(404, 'UserNotFound', 'u2011', '', '');
            return $responseBuilder;
        }

        // Authorize user show action
        if (!$this->errorHandler->hasErrors() && !$this->authorization->canShow($userModel)) {
            $this->errorHandler->addError(401, 'UserNotAuthorized', 'u2014', '', '');
        }

        //ok next step set the responseData
        $responseData = $this->factory->responseData()->user()->setFromUserModel($userModel);
        $responseBuilder->setResponse($responseData);

        return $responseBuilder;
    }

    /**
     * @param RequestDataUser $requestModel
     * @return ModelUser|null
     */
    private function getUserModel(RequestDataUser $requestModel)
    {
        if ($identifier = $requestModel->getIdentifier()) {
            return $this->database->getUserByIdentifier($identifier);
        }
        if ($guid = $requestModel->getGuid()) {
            return $this->database->getUserByGuid($guid);
        }
        if ($username = $requestModel->getUsername()) {
            return $this->database->getUserByUsername($username);
        }
        if ($email = $requestModel->getEmail()) {
            return $this->database->getUserByEmail($email);
        }
        return null;
    }
}
