<?php

/**
 * Class UserType
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Services\User\Update\Request\Version3;

use JMS\Serializer\Annotation as Serializer;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Class UserType
 *
 * @Serializer\XmlRoot("UserType")
 *
 * @copyright   2016 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 * @package     Sso\WebserviceBundle\Services\User\Update\Request\Version3
 */
class ValueUserType
{
    /**
     * @Serializer\Exclude
     * @var string
     */
    protected $identifier;

    /**
     *
     * @Assert\Email()
     *
     * @Serializer\SerializedName("Email")
     * @Serializer\Type("string")
     * @var string|null
     */
    protected $email;

    /**
     * @Assert\Regex(
     *     pattern="/\s/",
     *     match=false,
     *     message="The username contains whitespace characters which is not allowed"
     * )
     * @Assert\Length(
     *      min = 5,
     *      max = 240,
     *      minMessage = "Username must at least be {{ limit }} characters long",
     *      maxMessage = "Username can not exceed {{ limit }} characters"
     * )
     *
     * @Serializer\SerializedName("Username")
     * @Serializer\Type("string")
     * @var string
     */
    protected $username;

    /**
     * @Serializer\SerializedName("Firstname")
     * @Serializer\Type("string")
     * @var string
     */
    protected $firstname;

    /**
     * @Serializer\SerializedName("Lastname")
     * @Serializer\Type("string")
     * @var string
     */
    protected $lastname;

    /**
     * @Serializer\SerializedName("Password")
     * @Serializer\Type("string")
     * @var string
     */
    protected $password;

    /**
     * @Assert\Regex(
     *     pattern="/(0|1)/",
     *     match=true,
     *     message="The Active Flag can only contain 0 or 1"
     * )
     * @Serializer\SerializedName("Active")
     * @Serializer\Type("integer")
     * @var integer
     */
    protected $active;

    /**
     * @Serializer\SerializedName("Deleted")
     * @Serializer\Type("integer")
     * @var boolean
     */
    protected $deleted;

    /**
     * @Serializer\SerializedName("LastLoginTime")
     * @Serializer\Type("DateTime")
     * @var \DateTime
     */
    protected $lastLoginTime;

    /**
     * @Serializer\SerializedName("AuthId")
     * @Serializer\Type("string")
     * @var string
     */
    protected $authId;

    /**
     * @Serializer\SerializedName("LdapSearchAttributes")
     * @Serializer\Type("string")
     * @var string
     */
    protected $ldapSearchAttributes;

    /**
     * @Serializer\SerializedName("LdapSearchValue")
     * @Serializer\Type("string")
     * @var string
     */
    protected $ldapSearchValue;

    /**
     * @Assert\Regex(
     *     pattern="/(0|1)/",
     *     match=true,
     *     message="The MfaEnabled Flag can only contain 0 or 1"
     * )
     * @Serializer\SerializedName("MfaEnabled")
     * @Serializer\Type("integer")
     * @var integer
     */
    protected $mfaEnabled;

    /**
     * @Serializer\SerializedName("MfaRecreate")
     * @Serializer\Type("integer")
     * @var integer
     */
    protected $mfaRecreate;

    /**
     * @Serializer\SerializedName("ActivateTokenRecreate")
     * @Serializer\Type("integer")
     * @Serializer\Accessor(setter="setActivateTokenRecreate", getter="getActivateTokenRecreateAsInt")
     * @var boolean
     */
    protected $activateTokenRecreate;

    /**
     * @return string|null
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @return string
     */
    public function getUsername()
    {
        return $this->username;
    }

    /**
     * @return string
     */
    public function getFirstname()
    {
        return $this->firstname;
    }

    /**
     * @return string
     */
    public function getLastname()
    {
        return $this->lastname;
    }

    /**
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * @return boolean
     */
    public function isActive()
    {
        return $this->active;
    }

    /**
     * @return boolean
     */
    public function isDeleted()
    {
        return $this->deleted;
    }

    /**
     * @return \DateTime
     */
    public function getLastLoginTime()
    {
        return $this->lastLoginTime;
    }

    /**
     * @return string
     */
    public function getAuthId()
    {
        return $this->authId;
    }

    /**
     * @return string
     */
    public function getLdapSearchAttributes()
    {
        return $this->ldapSearchAttributes;
    }

    /**
     * @return string
     */
    public function getLdapSearchValue()
    {
        return $this->ldapSearchValue;
    }

    /**
     * @return boolean
     */
    public function isMfaEnabled()
    {
        return $this->mfaEnabled;
    }

    /**
     * @return boolean
     */
    public function isMfaRecreate()
    {
        return $this->mfaRecreate;
    }

    /**
     * @return bool
     */
    public function isActivateTokenRecreate()
    {
        return $this->activateTokenRecreate;
    }

    /**
     * @return int
     */
    public function getActivateTokenRecreateAsInt()
    {
        return $this->activateTokenRecreate ? 1 : 0;
    }

    /**
     * @param bool $activateTokenRecreate
     */
    public function setActivateTokenRecreate($activateTokenRecreate)
    {
        $this->activateTokenRecreate = (bool)$activateTokenRecreate;
    }

    /**
     * @return string
     */
    public function getIdentifier()
    {
        return $this->identifier;
    }

    /**
     * @param string $identifier
     * @return ValueUserType
     */
    public function setIdentifier($identifier)
    {
        $this->identifier = $identifier;
        return $this;
    }
}
