<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Services\User\Update\Response\Version5;

use Sso\WebserviceBundle\ErrorHandler\ErrorHandlerInterface;
use Sso\WebserviceBundle\Services\User\Update\ResponseBuilderInterface;
use Sso\WebserviceBundle\Services\ResponseBuilderInterface as GlobalResponseBuilderInterface;
use Sso\WebserviceBundle\Services\User\Update\ResponseData\User;
use JMS\Serializer\SerializerInterface;
use Sso\WebserviceBundle\Services\User\Update\ResponseData\User as UserData;

/**
 * Class ResponseBuilder
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\Services\User\Update\Response\Version5
 */
final class ResponseBuilder implements ResponseBuilderInterface, GlobalResponseBuilderInterface
{
    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var Factory
     */
    private $factory;

    /**
     * @var ErrorHandlerInterface
     */
    private $errorHandler;

    /**
     * @var UserData
     */
    private $responseData;

    /**
     * @var array
     */
    private $userTypeMapping = array(
        'getGuid' => 'guid',
        'getEmail' => 'email',
        'getUsername' => 'username',
        'getFirstname' => 'firstname',
        'getLastname' => 'lastname',
        'isActive' => 'active',
        'getAuthId' => 'authId',
        'getLdapSearchAttributes' => 'ldapSearchAttributes',
        'getLdapSearchValue' => 'ldapSearchValue',
        'isDeleted' => 'deleted',
        'getDeletedAt' => 'deletedAt',
        'getCreatedAt' => 'createdAt',
        'getUpdatedAt' => 'updatedAt',
        'getLastLoginAt' => 'lastLoginAt',
        'getLoginFails' => 'loginFails',
        'isMfaEnabled' => 'mfaEnabled',
        'isMfaRecreate' => 'mfaRecreate',
        'getPasswordPolicy' => 'passwordPolicy',
        'getLastPasswordChange' => 'lastPasswordChange',
        'getPasswordExpired' => 'passwordExpired',
        'getPasswordEncryptType' => 'passwordEncryptType',
        'getUserGroups' => 'userGroups',
    );

    /**
     * ResponseBuilder constructor.
     * @param SerializerInterface $serializer
     * @param Factory $factory
     * @param ErrorHandlerInterface $errorHandler
     */
    public function __construct(SerializerInterface $serializer, Factory $factory, ErrorHandlerInterface $errorHandler)
    {
        $this->serializer = $serializer;
        $this->factory = $factory;
        $this->errorHandler = $errorHandler;
    }

    /**
     * @param UserData $user
     */
    public function setResponse(User $user)
    {
        $this->responseData = $user;
    }

    /**
     * @return boolean
     */
    public function hasErrors()
    {
        return $this->errorHandler->hasErrors();
    }

    /**
     * @param string $format xml|json
     * @return string Response body
     */
    public function build($format)
    {
        $response = $this->errorHandler->hasErrors() ? $this->getErrorResponse() : $this->getSuccessResponse();

        return $this->serializer->serialize($response, $format);
    }

    /**
     * @return Response
     */
    private function getErrorResponse()
    {
        $response = $this->factory->response();
        $response->setErrors($this->errorHandler->getResponseObject());
        $response->setTrackid(null);
        $response->setDate(null);

        return $response;
    }

    /**
     * @return Response
     */
    private function getSuccessResponse()
    {
        $userType = $this->factory->userType();
        $userType->setFromArray($this->mapUserType($this->responseData));

        if (!$this->responseData->isActive() &&
            $this->responseData->getActivateToken() &&
            $this->responseData->getActivateTokenExpire()
        ) {
            $userType->setActivateToken($this->responseData->getActivateToken());
            $userType->setActivateTokenExpire($this->responseData->getActivateTokenExpire());
        }

        $update = $this->factory->update();
        $update->setUserType($userType);
        $update->setStatus('Success');
        $user = $this->factory->user();

        $user->addUpdate($update);

        $response = $this->factory->response();
        $response->setCode(200);
        $response->setScriptTimeSec($this->factory->scriptTimeSeconds());
        $response->setUser($user);

        return $response;
    }

    /**
     * @param UserData $user
     * @return array
     */
    private function mapUserType(UserData $user)
    {
        $userType = array();
        foreach ($this->userTypeMapping as $getter => $attributeName) {
            $userType[$attributeName] = $user->$getter();
        }

        return $userType;
    }
}
