<?php

/**
 * Class Key
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Services\UserSearch\Advanced\Request\Version3;

use Symfony\Component\Validator\Constraints as Assert;
use JMS\Serializer\Annotation as Serializer;

/**
 * Class Key
 *
 * @Serializer\XmlRoot("Filter")
 *
 * @copyright  2016 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Sso\WebserviceBundle\Services\UUserSearch\Advanced\Request\Version3;
 */
final class Filter
{
    /**
     * @Assert\NotBlank()
     * @Assert\Regex(
     *     pattern="/(^EQ$|^NOT EQ$|^GT$|^GTE$|^LT$|^LTE$|^LIKE$|^NOT LIKE$|^REGEX$|^IN$)/",
     *     match=true,
     *     message="Strategy must be (EQ|NOT EQ|GT|GTE|LT|LTE|LIKE|NOT LIKE|REGEX|IN)"
     * )
     *
     * @Serializer\SerializedName("Strategy")
     * @Serializer\Type("string")
     * @var string
     */
    private $strategy;

    /**
     * @Assert\NotBlank()
     * @Assert\Regex(
     *     pattern="/(^AND NOT$|^AND$|^OR$)/",
     *     match=true,
     *     message="Operator must be (AND|OR|AND NOT)"
     * )
     * @Serializer\SerializedName("Operator")
     * @Serializer\Type("string")
     * @var string
     */
    private $operator;

    /**
     * @Assert\Valid
     * @Serializer\SerializedName("UserType")
     * @Serializer\Type("Sso\WebserviceBundle\Services\UserSearch\Advanced\Request\Version3\UserType")
     * @var UserType
     */
    private $userType;

    /**
     * @Assert\Valid
     *
     * @Serializer\SerializedName("ApplicationType")
     * @Serializer\Type("Sso\WebserviceBundle\Services\UserSearch\Advanced\Request\Version3\ApplicationType")
     * @var ApplicationType
     */
    private $applicationType;

    /**
     * @Assert\Valid
     *
     * @Serializer\SerializedName("Filters")
     * @Serializer\Type("Sso\WebserviceBundle\Services\UserSearch\Advanced\Request\Version3\Filters")
     * @var Filters
     */
    private $filters;

    /**
     * @return string
     */
    public function getOperator()
    {
        return $this->operator;
    }

    /**
     * @param string $operator
     * @return Filter
     */
    public function setOperator($operator)
    {
        $this->operator = $operator;

        return $this;
    }

    /**
     * @return string
     */
    public function getStrategy()
    {
        return $this->strategy;
    }

    /**
     * @param string $strategy
     * @return Filter
     */
    public function setStrategy($strategy)
    {
        $this->strategy = $strategy;

        return $this;
    }

    /**
     * @return UserType
     */
    public function getUserType()
    {
        return $this->userType;
    }

    /**
     * @param UserType $userType
     * @return Filter
     */
    public function setUserType(UserType $userType)
    {
        $this->userType = $userType;

        return $this;
    }

    /**
     * @return ApplicationType
     */
    public function getApplicationType()
    {
        return $this->applicationType;
    }

    /**
     * @param ApplicationType $applicationType
     * @return Filter
     */
    public function setApplicationType(ApplicationType $applicationType)
    {
        $this->applicationType = $applicationType;

        return $this;
    }

    /**
     * @return Filters
     */
    public function getFilters()
    {
        return $this->filters;
    }

    /**
     * @param Filters $filters
     * @return Filter
     */
    public function setFilters(Filters $filters)
    {
        $this->filters = $filters;

        return $this;
    }

    /**
     * @Assert\IsTrue(message="The regular expression is not valid!")
     */
    public function isRegularExpressionValid()
    {
        if ('REGEX' !== $this->strategy) {
            return true;
        }

        if (null !== $this->applicationType && !$this->applicationType->isRegularExpressionValid()) {
            return false;
        }

        if (null !== $this->userType && !$this->userType->isRegularExpressionValid()) {
            return false;
        }

        return true;
    }
}
