<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Sso\WebserviceBundle\Validator\Constraints;

use Sso\WebserviceBundle\Services\UserApplicationAttribute\Add\Request\UserTypeInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Class UserTypeIdentifierValidator
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Sso\WebserviceBundle\Validator\Constraints
 */
class UserTypeIdentifierValidator extends ConstraintValidator
{
    /**
     * Checks if the passed value is valid.
     *
     * @param mixed $value The value that should be validated
     * @param Constraint $constraint The constraint for the validation
     */
    public function validate($value, Constraint $constraint)
    {
        if (null === $value) {
            return;
        }

        if (!$value instanceof UserTypeInterface) {
            throw new UnexpectedTypeException($value, UserTypeInterface::class);
        }

        if ($value->hasUsername()) {
            $this->validateUsername($value, $constraint);
        } elseif ($value->hasIdentifier()) {
            $this->validateIdentifier($value, $constraint);
        } elseif ($value->hasGuid()) {
            $this->validateGuid($value, $constraint);
        } elseif ($value->hasEmail()) {
            $this->validateEmail($value, $constraint);
        } else {
            $this->context->buildViolation($constraint->message)
                ->setParameter('{{ properties }}', 'NONE SET')
                ->addViolation();
        }
    }

    /**
     * @param UserTypeInterface $userType
     * @param Constraint $constraint
     */
    private function validateUsername(UserTypeInterface $userType, Constraint $constraint)
    {
        if (0 === strlen($userType->getUsername())) {
            $this->context->buildViolation($constraint->notBlankMessage)
                ->setParameter('{{ property }}', 'Username')
                ->addViolation();

        }
        $setProperties = ['Username'];
        $userType->hasGuid() ? $setProperties[] = 'Guid' : null;
        $userType->hasEmail() ? $setProperties[] = 'Email' : null;
        $userType->hasIdentifier() ? $setProperties[] = 'Identifier' : null;
        if (1 < count($setProperties)) {
            $this->context->buildViolation($constraint->message)
                ->setParameter('{{ properties }}', join(', ', $setProperties))
                ->addViolation();
        }
    }

    /**
     * @param UserTypeInterface $userType
     * @param Constraint $constraint
     */
    private function validateEmail(UserTypeInterface $userType, Constraint $constraint)
    {
        if (0 === strlen($userType->getEmail())) {
            $this->context->buildViolation($constraint->notBlankMessage)
                ->setParameter('{{ property }}', 'Email')
                ->addViolation();

        }
        $setProperties = ['Email'];
        $userType->hasGuid() ? $setProperties[] = 'Guid' : null;
        $userType->hasUsername() ? $setProperties[] = 'Username' : null;
        $userType->hasIdentifier() ? $setProperties[] = 'Identifier' : null;
        if (1 < count($setProperties)) {
            $this->context->buildViolation($constraint->message)
                ->setParameter('{{ properties }}', join(', ', $setProperties))
                ->addViolation();
        }
    }

    /**
     * @param UserTypeInterface $userType
     * @param Constraint $constraint
     */
    private function validateGuid(UserTypeInterface $userType, Constraint $constraint)
    {
        if (0 === strlen($userType->getGuid())) {
            $this->context->buildViolation($constraint->notBlankMessage)
                ->setParameter('{{ property }}', 'Guid')
                ->addViolation();

        }
        $setProperties = ['Guid'];
        $userType->hasEmail() ? $setProperties[] = 'Email' : null;
        $userType->hasUsername() ? $setProperties[] = 'Username' : null;
        $userType->hasIdentifier() ? $setProperties[] = 'Identifier' : null;
        if (1 < count($setProperties)) {
            $this->context->buildViolation($constraint->message)
                ->setParameter('{{ properties }}', join(', ', $setProperties))
                ->addViolation();
        }
    }

    /**
     * @param UserTypeInterface $userType
     * @param Constraint $constraint
     */
    private function validateIdentifier(UserTypeInterface $userType, Constraint $constraint)
    {
        if (0 === strlen($userType->getIdentifier())) {
            $this->context->buildViolation($constraint->notBlankMessage)
                ->setParameter('{{ property }}', 'Identifier')
                ->addViolation();

        }
        $setProperties = ['Identifier'];
        $userType->hasEmail() ? $setProperties[] = 'Email' : null;
        $userType->hasUsername() ? $setProperties[] = 'Username' : null;
        $userType->hasGuid() ? $setProperties[] = 'Guid' : null;
        if (1 < count($setProperties)) {
            $this->context->buildViolation($constraint->message)
                ->setParameter('{{ properties }}', join(', ', $setProperties))
                ->addViolation();
        }
    }
}