<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 */

namespace LifeStyle\DH\AccessControlBundle\EventListener;

use LifeStyle\DH\AccessControlBundle\Authorization\UserScopesBuilder;
use LifeStyle\DH\AccessControlBundle\Exception\InvalidConfigurationException;
use LifeStyle\DH\AccessControlBundle\Exception\InvalidRequestException;
use LifeStyle\DH\AccessControlBundle\Repository\ObjectServiceRepository;
use Psr\Log\LoggerInterface;
use Sso\AccessBundle\Authorization\TemporaryTokenAuthorizationInterface;
use Sso\WebserviceBundle\Event\UserAuthorizationEvent;
use Sso\WebserviceBundle\Security\Authentication\Token\WsFirewallToken;
use Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface;

/**
 * Class UserAuthorizationListener
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       https://www.life-style.de
 * @package LifeStyle\DH\AccessControlBundle\EventListener
 */
class UserAuthorizationListener
{
    /**
     * @var TokenStorageInterface
     */
    private $tokenStorageInterface;

    /**
     * @var TemporaryTokenAuthorizationInterface
     */
    private $temporaryTokenAuthorization;

    /**
     * @var ObjectServiceRepository
     */
    private $objectServiceRepository;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var UserScopesBuilder
     */
    private $userScopesBuilder;

    /**
     * UserAuthorizationListener constructor.
     * @param TokenStorageInterface $tokenStorageInterface
     * @param TemporaryTokenAuthorizationInterface $temporaryTokenAuthorization
     * @param ObjectServiceRepository $objectServiceRepository
     * @param LoggerInterface $logger
     * @param UserScopesBuilder $userScopesBuilder
     */
    public function __construct(
        TokenStorageInterface $tokenStorageInterface,
        TemporaryTokenAuthorizationInterface $temporaryTokenAuthorization,
        ObjectServiceRepository $objectServiceRepository,
        LoggerInterface $logger,
        UserScopesBuilder $userScopesBuilder
    ) {
        $this->tokenStorageInterface = $tokenStorageInterface;
        $this->temporaryTokenAuthorization = $temporaryTokenAuthorization;
        $this->objectServiceRepository = $objectServiceRepository;
        $this->logger = $logger;
        $this->userScopesBuilder = $userScopesBuilder;
    }

    /**
     * Checks if the current application user consuming the webservice
     * is allowed to update/show/delete the user from request
     *
     * @param UserAuthorizationEvent $event
     * @throws InvalidConfigurationException
     */
    public function onUserChange(UserAuthorizationEvent $event)
    {
        if (!$this->hasRestrictedAccess()) {
            return;
        }

        $applicationUserScopes = $this->userScopesBuilder->build();

        // Global HR manager has no restrictions
        if ($applicationUserScopes->isGlobalHrManager()) {
            return;
        }

        // User is not allowed to view/edit/delete himself
        $user = $event->getUser();
        if ($user->getId() === $applicationUserScopes->getUserId()) {
            $this->logger->info('Application user is not allowed to change his own user account.');
            $event->unAuthorize();
            return;
        }

        // If user has been added by the same application, user does have granted access on this user
        if (in_array($user->getId(), $this->temporaryTokenAuthorization->getAuthorizedUserIds())) {
            return;
        }

        try {
            $applicationUserScopeIds = $this->objectServiceRepository->findObjectIdsByScopeIds(
                $applicationUserScopes->getScopeIds()
            );
        } catch (InvalidRequestException $exception) {
            // We don't want errors from object-ws to bubble up
            $this->logger->debug(sprintf(
                'Error while receiving authorization details from object-ws! %s',
                $exception->getMessage()
            ));
            $applicationUserScopeIds = [];
        }

        $userScopeIds = $this->userScopesBuilder->build($user)->getScopeIds();

        $isAuthorized = 0 < count(array_intersect($userScopeIds, $applicationUserScopeIds));
        if (!$isAuthorized) {
            $this->logger->info(sprintf(
                'Application user is not allowed to change user (id: %s) because of scope restrictions.',
                $user->getId()
            ));
            $event->unAuthorize();
        }
    }

    /**
     * @return bool
     */
    private function hasRestrictedAccess(): bool
    {
        return
            ($token = $this->tokenStorageInterface->getToken()) &&
            $token instanceof WsFirewallToken &&
            $token->accessRestricted;
    }
}
