<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Sulu\Swoox\ContentFilterBundle\Content\Types;

use LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\Form\SwooxFilterFormBuilderInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\SearchHandlerInterface;
use Sulu\Component\Content\Compat\PropertyInterface;
use Sulu\Component\Content\SimpleContentType;
use Sulu\Component\DocumentManager\Exception\DocumentManagerException;
use Sulu\Component\DocumentManager\Exception\DocumentNotFoundException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class SwooxContentFilter
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package LifeStyle\Sulu\Swoox\ContentFilterBundle\Content\Types
 */
class SwooxContentFilter extends SimpleContentType
{
    /**
     * @var ContainerInterface
     */
    private $container;

    /**
     * SwooxContentFilter constructor.
     * @param ContainerInterface $container
     * @param null $template
     */
    public function __construct(
        ContainerInterface $container,
        $template
    ) {
        parent::__construct('SwooxContentFilter', '');
        $this->container = $container;
        $this->template = $template;
    }

    /**
     * Returns a template to render a form.
     *
     * @return string
     */
    public function getTemplate()
    {
        return $this->template;
    }

    /**
     * {@inheritdoc}
     */
    public function getDefaultParams(PropertyInterface $property = null)
    {
        $request = $this->container->get('request_stack')->getCurrentRequest();
        $requestQuery = $request->query;

        $id = $requestQuery->get('uuid');
        $language = $requestQuery->get('language');
        $ghostContent = $requestQuery->get('ghost-content', false);
        $template = $requestQuery->get('template');
        $webspaceName = $requestQuery->get('webspace');

        $document = $this->findDocument($id, $language, $ghostContent, $template);
        $node = null !== $document ? $this->findNode($id) : null;
        $prefix = null !== $document ? $this->findLocalizedPrefix() : false;
        $internalPrefix = ''; // Whaaaaat?

        $swooxLocale = $language;
        $swooxQueryParams = [];
        if (null !== $document && null !== $node && false !== $prefix) {
            $locale = $this->container->get('sulu_document_manager.document_inspector')->getLocale($document);
            $nodeHelper = $this->container->get('life_style_sulu_swoox_content_filter.content_extension.swoox_content_filter_extension');
            $nodeHelper->setLanguageCode($locale, $prefix, $internalPrefix);
            $swooxQuery = $nodeHelper->load($node, $webspaceName, $locale);
            parse_str($swooxQuery, $swooxQueryParams);
        }

        $filterSearchResponse = $this->container->get(SearchHandlerInterface::class)->filterSearch($swooxQueryParams, $swooxLocale);
        $form = $this->container->get(SwooxFilterFormBuilderInterface::class)->buildFilterForm($filterSearchResponse);

        return [
            'filterForm' => $form->createView(),
        ];
    }

    /**
     * @return bool|string
     */
    private function findLocalizedPrefix()
    {
        try {
            return $this->container->get('sulu_document_manager.namespace_registry')->getPrefix('extension_localized');
        } catch (DocumentManagerException $exception) {
            return false;
        }
    }

    /**
     * @param string $id
     * @param string $language
     * @param string $ghostContent
     * @param string $template
     * @return object|null
     */
    private function findDocument($id, $language, $ghostContent, $template)
    {
        $documentManager = $this->container->get('sulu_document_manager.document_manager');
        try {
            $document = $documentManager->find(
                $id,
                $language,
                [
                    'load_ghost_content' => $ghostContent,
                    'structure_type' => $template,
                ]
            );
        } catch (DocumentNotFoundException $exception) {
            $document = null;
        } catch (DocumentManagerException $exception) {
            $document = null;
        }

        return $document;
    }

    /**
     * @param string $id
     * @return \PHPCR\NodeInterface|null
     */
    private function findNode($id)
    {
        try {
            $node = $this->container->get('sulu_document_manager.node_manager')->find($id);
        } catch (DocumentNotFoundException $exception) {
            $node = null;
        }

        return $node;
    }
}
