<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Sulu\Swoox\ContentFilterBundle\AdminHelper;

use LifeStyle\Sulu\Swoox\ContentFilterBundle\Content\Extension\SwooxContentFilterExtension;
use Sulu\Component\DocumentManager\Exception\DocumentManagerException;
use Sulu\Component\DocumentManager\Exception\DocumentNotFoundException;
use Symfony\Component\HttpFoundation\Request;
use Sulu\Component\DocumentManager\DocumentInspector;
use Sulu\Component\DocumentManager\NamespaceRegistry;
use Sulu\Component\DocumentManager\DocumentManager;
use Sulu\Component\DocumentManager\NodeManager;

/**
 * Class RequestParser
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package LifeStyle\Sulu\Swoox\ContentFilterBundle\AdminHelper
 */
class RequestParser
{
    /**
     * @var DocumentInspector
     */
    private $documentInspector;

    /**
     * @var DocumentManager
     */
    private $documentManager;

    /**
     * @var NodeManager
     */
    private $documentNodeManager;

    /**
     * @var NamespaceRegistry
     */
    private $documentNamespaceRegistry;

    /**
     * @var SwooxContentFilterExtension
     */
    private $swooxContentFilterExtension;

    /**
     * RequestParser constructor.
     * @param DocumentInspector $documentInspector
     * @param DocumentManager $documentManager
     * @param NodeManager $documentNodeManager
     * @param NamespaceRegistry $documentNamespaceRegistry
     * @param SwooxContentFilterExtension $swooxContentFilterExtension
     */
    public function __construct(
        DocumentInspector $documentInspector,
        DocumentManager $documentManager,
        NodeManager $documentNodeManager,
        NamespaceRegistry $documentNamespaceRegistry,
        SwooxContentFilterExtension $swooxContentFilterExtension
    ) {
        $this->documentInspector = $documentInspector;
        $this->documentManager = $documentManager;
        $this->documentNodeManager = $documentNodeManager;
        $this->documentNamespaceRegistry = $documentNamespaceRegistry;
        $this->swooxContentFilterExtension = $swooxContentFilterExtension;
    }

    /**
     * This is where the magic happens! Pull out request parameters from PHPCR content
     *
     * @param Request $request
     * @return array
     */
    public function parseRequest(Request $request): array
    {
        $requestQuery = $request->query;

        $id = $requestQuery->get('uuid');
        $language = $requestQuery->get('language');
        $ghostContent = $requestQuery->get('ghost-content', false);
        $template = $requestQuery->get('template');
        $webspaceName = $requestQuery->get('webspace');

        $document = $this->findDocument($id, $language, $ghostContent, $template);
        $node = null !== $document ? $this->findNode($id) : null;
        $prefix = null !== $document ? $this->findLocalizedPrefix() : false;
        $internalPrefix = ''; // Whaaaaat?

        $swooxQueryParams = [];
        if (null !== $document && null !== $node && false !== $prefix) {
            $locale = $this->documentInspector->getLocale($document);
            $this->swooxContentFilterExtension->setLanguageCode($locale, $prefix, $internalPrefix);
            $swooxQuery = $this->swooxContentFilterExtension->load($node, $webspaceName, $locale);
            parse_str($swooxQuery, $swooxQueryParams);
        }

        return $swooxQueryParams;
    }

    /**
     * @return bool|string
     */
    private function findLocalizedPrefix()
    {
        try {
            return $this->documentNamespaceRegistry->getPrefix('extension_localized');
        } catch (DocumentManagerException $exception) {
            return false;
        }
    }

    /**
     * @param string $id
     * @param string $language
     * @param string $ghostContent
     * @param string $template
     * @return object|null
     */
    private function findDocument($id, $language, $ghostContent, $template)
    {
        try {
            $document = $this->documentManager->find(
                $id,
                $language,
                [
                    'load_ghost_content' => $ghostContent,
                    'structure_type' => $template,
                ]
            );
        } catch (DocumentNotFoundException $exception) {
            $document = null;
        } catch (DocumentManagerException $exception) {
            $document = null;
        }

        return $document;
    }

    /**
     * @param string $id
     * @return \PHPCR\NodeInterface|null
     */
    private function findNode($id)
    {
        try {
            $node = $this->documentNodeManager->find($id);
        } catch (DocumentNotFoundException $exception) {
            $node = null;
        }

        return $node;
    }
}
