<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sulu\TaxonBundle\MessageHandler;

use Doctrine\ORM\EntityManagerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\RabbitMq\Consumer\Handler\HandlerInterface;
use Lifestyle\Sulu\TaxonBundle\Database\ReconnectTrait;
use Lifestyle\Sulu\TaxonBundle\Entity\Factory;
use Lifestyle\Sulu\TaxonBundle\Entity\Taxon;
use Lifestyle\Sulu\TaxonBundle\Mapper\MessageMapper;
use Lifestyle\Sulu\TaxonBundle\Repository\TaxonRepository;

/**
 * Class TaxonPublishHandler
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sulu\TaxonBundle\MessageHandler
 */
class TaxonPublishHandler implements HandlerInterface
{
    use ReconnectTrait;

    /**
     * @var MessageMapper
     */
    private $mapper;

    /**
     * @var Factory
     */
    private $factory;

    /**
     * @var EntityManagerInterface
     */
    private $objectManager;

    /**
     * TaxonPublishHandler constructor.
     * @param MessageMapper $mapper
     * @param Factory $factory
     * @param EntityManagerInterface $objectManager
     */
    public function __construct(MessageMapper $mapper, Factory $factory, EntityManagerInterface $objectManager)
    {
        $this->mapper = $mapper;
        $this->factory = $factory;
        $this->objectManager = $objectManager;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool Should return true, if data has been processed
     * @throws \Doctrine\ORM\ORMException
     * @throws \Lifestyle\Sulu\TaxonBundle\Exception\DatabaseException
     */
    public function handle(DataCollectorInterface $dataCollector): bool
    {
        if (!$this->isResponsible($dataCollector)) {
            return false;
        }

        $this->objectManager = $this->getClearedEntityManager($this->objectManager);
        /** @var TaxonRepository $objectRepository */
        $objectRepository = $this->objectManager->getRepository(Taxon::class);

        $taxonId = $dataCollector->getItemValue('id');
        foreach ($dataCollector->getCollectedLocales() as $locale) {
            $taxon = $objectRepository->findOneByTaxonIdAndLocale($taxonId, $locale);
            if (null === $taxon) {
                $taxon = $this->factory->taxon();
                $this->objectManager->persist($taxon);
            }
            $this->mapper->map($taxon, $dataCollector, $locale);
        }
        $this->flush($this->objectManager);

        return true;
    }

    /**
     * @param DataCollectorInterface $dataCollector
     * @return bool
     */
    private function isResponsible(DataCollectorInterface $dataCollector): bool
    {
        return 0 === strcasecmp($dataCollector->getItemValue('event'), 'taxon.publish');
    }
}
