<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sulu\TaxonBundle\Content\Types;

use Doctrine\ORM\EntityManagerInterface;
use Lifestyle\Sulu\TaxonBundle\Entity\Taxon;
use Lifestyle\Sulu\TaxonBundle\Mapper\OptionSelectMapper;
use Lifestyle\Sulu\TaxonBundle\MessageProducer\TaxonUpdateProducer;
use PHPCR\NodeInterface;
use Sulu\Component\Content\Compat\PropertyInterface;
use Sulu\Component\Content\SimpleContentType;
use Sulu\Component\DocumentManager\DocumentManager;
use Sulu\Component\DocumentManager\Exception\DocumentManagerException;
use Sulu\Component\Webspace\Manager\WebspaceManagerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Class TaxonSelect
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package Lifestyle\Sulu\TaxonBundle\Content\Types
 */
class TaxonSelect extends SimpleContentType
{
    /**
     * @var RequestStack
     */
    private $requestStack;

    /**
     * @var OptionSelectMapper
     */
    private $mapper;

    /**
     * @var EntityManagerInterface
     */
    private $objectManager;

    /**
     * @var DocumentManager
     */
    private $documentManager;

    /**
     * @var WebspaceManagerInterface
     */
    private $webspaceManager;

    /**
     * @var string
     */
    private $template;

    /**
     * @var string
     */
    private $environment;

    /**
     * @var TaxonUpdateProducer
     */
    private $taxonUpdater;

    /**
     * TaxonSelect constructor.
     * @param RequestStack $requestStack
     * @param OptionSelectMapper $mapper
     * @param EntityManagerInterface $objectManager
     * @param DocumentManager $documentManager
     * @param WebspaceManagerInterface $webspaceManager
     * @param string $template
     * @param string $environment
     * @param TaxonUpdateProducer $taxonUpdater
     */
    public function __construct(
        RequestStack $requestStack,
        OptionSelectMapper $mapper,
        EntityManagerInterface $objectManager,
        DocumentManager $documentManager,
        WebspaceManagerInterface $webspaceManager,
        string $template,
        string $environment,
        TaxonUpdateProducer $taxonUpdater
    ) {
        parent::__construct('TaxonSelect', '');
        $this->requestStack = $requestStack;
        $this->mapper = $mapper;
        $this->objectManager = $objectManager;
        $this->documentManager = $documentManager;
        $this->webspaceManager = $webspaceManager;
        $this->template = $template;
        $this->environment = $environment;
        $this->taxonUpdater = $taxonUpdater;
    }

    /**
     * Returns a template to render a form.
     *
     * @return string
     */
    public function getTemplate()
    {
        return $this->template;
    }

    public function write(
        NodeInterface $node,
        PropertyInterface $property,
        $userId,
        $webspaceKey,
        $languageCode,
        $segmentKey
    ) {
        parent::write($node, $property, $userId, $webspaceKey, $languageCode, $segmentKey);

        $taxonId = $property->getValue();
        $taxon = $this->objectManager->getRepository(Taxon::class)->find($taxonId);
        if (null !== $taxon) {
            $request = $this->requestStack->getCurrentRequest();
            $documentUuid = $request->attributes->get('uuid');
            try {
                $resource = $this->documentManager->find($documentUuid, $languageCode)->getResourceSegment();
                $path = preg_replace('#^https?://[^/]*/#', '/', $this->webspaceManager->findUrlByResourceLocator(
                    $resource,
                    $this->environment,
                    $languageCode,
                    $webspaceKey
                ));
            } catch (DocumentManagerException $exception) {
                $path = '';
            }
            $this->taxonUpdater->send($taxon, $path);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getDefaultParams(PropertyInterface $property = null)
    {
        $request = $this->requestStack->getCurrentRequest();
        $locale = $request->query->get('language');

        // Set taxons for mapper
        $this->mapper->setTaxons($this->objectManager->getRepository(Taxon::class)->findByLocale($locale));

        // Admin
        return [
            'select_options' => $this->mapper->map(),
        ];
    }
}
