<?php

/*
 * This file is part of Sulu.
 *
 * (c) MASSIVE ART WebServices GmbH
 *
 * This source file is subject to the MIT license that is bundled
 * with this source code in the file LICENSE.
 */

namespace LifeStyle\Sulu\SwooxMassiveSearchBundle\Controller;

use LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\ConfigurationInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\Form\SwooxFilterFormBuilderInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\SearchHandlerInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\View\Mapper\SwooxMapperChainInterface;
use Sulu\Bundle\WebsiteBundle\Resolver\ParameterResolverInterface;
use Sulu\Component\Rest\RequestParametersTrait;
use Sulu\Component\Webspace\Analyzer\RequestAnalyzerInterface;
use Symfony\Bundle\FrameworkBundle\Templating\EngineInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Knp\Component\Pager\PaginatorInterface;

/**
 * Class WebsiteSearchController
 * @package LifeStyle\Sulu\SwooxMassiveSearchBundle\Controller
 */
class WebsiteSearchController
{
    use RequestParametersTrait;

    /**
     * @var SearchHandlerInterface
     */
    private $searchHandler;

    /**
     * @var RequestAnalyzerInterface
     */
    private $requestAnalyzer;

    /**
     * @var ParameterResolverInterface
     */
    private $parameterResolver;

    /**
     * @var EngineInterface
     */
    private $engine;

    /**
     * @var PaginatorInterface
     */
    private $paginator;

    /**
     * @var SwooxFilterFormBuilderInterface
     */
    private $swooxFilterFormBuilder;

    /**
     * @var SwooxMapperChainInterface
     */
    private $swooxMapperChain;

    /**
     * @var ConfigurationInterface
     */
    private $swooxConfiguration;

    /**
     * WebsiteSearchController constructor.
     * @param SearchHandlerInterface $searchHandler
     * @param RequestAnalyzerInterface $requestAnalyzer
     * @param ParameterResolverInterface $parameterResolver
     * @param EngineInterface $engine
     * @param PaginatorInterface $paginator
     * @param SwooxFilterFormBuilderInterface $swooxFilterFormBuilder
     * @param SwooxMapperChainInterface $swooxMapperChain
     * @param ConfigurationInterface $swooxConfiguration
     */
    public function __construct(
        SearchHandlerInterface $searchHandler,
        RequestAnalyzerInterface $requestAnalyzer,
        ParameterResolverInterface $parameterResolver,
        EngineInterface $engine,
        PaginatorInterface $paginator,
        SwooxFilterFormBuilderInterface $swooxFilterFormBuilder,
        SwooxMapperChainInterface $swooxMapperChain,
        ConfigurationInterface $swooxConfiguration
    ) {
        $this->searchHandler = $searchHandler;
        $this->requestAnalyzer = $requestAnalyzer;
        $this->parameterResolver = $parameterResolver;
        $this->engine = $engine;
        $this->paginator = $paginator;
        $this->swooxFilterFormBuilder = $swooxFilterFormBuilder;
        $this->swooxMapperChain = $swooxMapperChain;
        $this->swooxConfiguration = $swooxConfiguration;
    }

    /**
     * @param Request $request
     * @return Response
     * @throws \Sulu\Component\Rest\Exception\MissingParameterException
     */
    public function queryAction(Request $request)
    {
        $keywords = $this->getRequestParameter($request, 'q', true);

        $locale = $this->requestAnalyzer->getCurrentLocalization()->getLocale();
        $webspace = $this->requestAnalyzer->getWebspace();

        $swooxQueryParams = $request->query->all();
        $swooxQueryParams['keywords'] = $keywords;

        $limit = $request->query->get('limit', $this->swooxConfiguration->getDefaultLimit());
        $swooxQueryParams['limit'] = $limit;

        $page = $request->query->get('page', 1);
        $swooxQueryParams['offset'] = ($page - 1) * $limit;

        unset(
            $swooxQueryParams['page'],
            $swooxQueryParams['q'],
            $swooxQueryParams['filter'],
            $swooxQueryParams['_token']
        );

        $responseDto = $this->searchHandler->standardSearch($swooxQueryParams, $locale);
        $mappedResponseDto = $this->swooxMapperChain->map($responseDto, $locale);

        $filterSearchResponse = $this->searchHandler->filterSearch($swooxQueryParams, $locale);

        // Build filter form
        $form = $this->swooxFilterFormBuilder->buildFilterForm($filterSearchResponse);
        $pagination = $this->paginator->paginate($mappedResponseDto, $page, $limit);

        return $this->engine->renderResponse(
            $webspace->getTemplate('search'),
            $this->parameterResolver->resolve(
                ['query' => $keywords, 'pagination' => $pagination, 'form' => $form->createView(), 'locale' => $locale],
                $this->requestAnalyzer
            )
        );
    }
}
