<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\Form;

use LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\Configuration;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormFactoryInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Form\Extension\Core\Type as FormType;
use Swoox\Api\Model\Response\Search\Filter\FilterResponse;

/**
 * Class SwooxFilterFormBuilder
 * @package LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\Form
 */
class SwooxFilterFormBuilder implements SwooxFilterFormBuilderInterface
{
    /**
     * @var FormFactoryInterface
     */
    private $formFactory;

    /**
     * @var Configuration
     */
    private $configuration;

    /**
     * FilterFormBuilder constructor.
     * @param FormFactoryInterface $formFactory
     * @param Configuration $configuration
     */
    public function __construct(FormFactoryInterface $formFactory, Configuration $configuration)
    {
        $this->formFactory = $formFactory;
        $this->configuration = $configuration;
    }

    /**
     * @param FilterResponse $filterResponse
     * @return FormInterface
     */
    public function buildFilterForm(FilterResponse $filterResponse): FormInterface
    {
        $form = $this->formFactory->createNamedBuilder(null);
        $form = $this->createValueArray($form, $filterResponse);
        $form = $this->createMultiSelectArray($form, $filterResponse);

        //add hidden keywords
        $form->add('q', FormType\HiddenType::class, ['data' => $filterResponse->getKeywords()]);
        $form->add('filter', FormType\SubmitType::class, array('label' => 'Filter'));

        return $form->getForm();
    }

    /**
     * @param FormBuilderInterface $form
     * @param FilterResponse $filterResponse
     * @return FormBuilderInterface
     */
    private function createValueArray(FormBuilderInterface $form, FilterResponse $filterResponse)
    {
        if (null === $filterResponse->getFilterValueArray()) {
            return $form;
        }

        foreach ($filterResponse->getFilterValueArray()->getFilters() as $valueFilter) {
            $choices = [];
            $data = null;
            foreach ($valueFilter->getValueArray()->getItems() as $value) {
                $choices[$value->getValue()] = $value->getClean();
                if ('yes' == $value->getSelected()) {
                    $data = $value->getClean();
                }
            }

            if ("none" !== $this->configuration->getFilterSelectSorting()) {
                // sort array in natural way
                ksort($choices, constant($this->configuration->getFilterSelectSorting()));
            }


            if($valueFilter->getName() != 'locale'){
                $form->add(
                    $valueFilter->getSlotClean() ? $valueFilter->getSlotClean() : $valueFilter->getSlot(),
                    FormType\ChoiceType::class,
                    [
                        'choices' => $choices,
                        'required' => false,
                        'data' => $data,
                        'placeholder' => 'filter_choice_placeholder',
                        'label' => $valueFilter->getName()
                    ]
                );
            }

        }
        return $form;
    }

    /**
     * @param FormBuilderInterface $form
     * @param FilterResponse $filterResponse
     * @return FormBuilderInterface
     */
    private function createMultiSelectArray(FormBuilderInterface $form, FilterResponse $filterResponse)
    {
        if (null === $filterResponse->getFilterMultiSelectArray()) {
            return $form;
        }

        foreach ($filterResponse->getFilterMultiSelectArray()->getFilters() as $valueFilter) {
            $choices = [];
            $data = [];
            foreach ($valueFilter->getValueArray()->getItems() as $value) {
                $choices[$value->getValue() . " (" . $value->getCount() . ")"] = $value->getClean();
                if ('yes' == $value->getSelected()) {
                    $data[] = $value->getClean();
                }
            }

            if ("none" !== $this->configuration->getFilterSelectSorting()) {
                // sort array in natural way
                ksort($choices, constant($this->configuration->getFilterSelectSorting()));
            }

            $form->add(
                $valueFilter->getSlotClean() ? $valueFilter->getSlotClean() : $valueFilter->getSlot(),
                FormType\ChoiceType::class,
                [
                    'choices' => $choices,
                    'multiple' => true,
                    'expanded' => true,
                    'required' => false,
                    'data' => $data,
                    'label' => $valueFilter->getName()
                ]
            );
        }
        return $form;
    }
}
