<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Sulu\SwooxMassiveSearchBundle\View;

use LifeStyle\Sulu\SwooxMassiveSearchBundle\Pagination\LsPaginationInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\ConfigurationInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\SearchHandlerInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\View\Mapper\SwooxMapperChainInterface;
use Knp\Component\Pager\PaginatorInterface;
use Knp\Component\Pager\Pagination\PaginationInterface;

/**
 * Class SearchEngine
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package LifeStyle\Sulu\SwooxMassiveSearchBundle\View
 */
class SearchEngine implements SearchEngineInterface
{
    /**
     * @var SearchHandlerInterface
     */
    private $searchHandler;

    /**
     * @var PaginatorInterface
     */
    private $paginator;

    /**
     * @var SwooxMapperChainInterface
     */
    private $swooxMapperChain;

    /**
     * @var ConfigurationInterface
     */
    private $swooxConfiguration;

    /**
     * SearchEngine constructor.
     * @param SearchHandlerInterface $searchHandler
     * @param PaginatorInterface $paginator
     * @param SwooxMapperChainInterface $swooxMapperChain
     * @param ConfigurationInterface $swooxConfiguration
     */
    public function __construct(
        SearchHandlerInterface $searchHandler,
        PaginatorInterface $paginator,
        SwooxMapperChainInterface $swooxMapperChain,
        ConfigurationInterface $swooxConfiguration
    ) {
        $this->searchHandler = $searchHandler;
        $this->paginator = $paginator;
        $this->swooxMapperChain = $swooxMapperChain;
        $this->swooxConfiguration = $swooxConfiguration;
    }

    /**
     * @param array $searchParams
     * @param string $locale
     * @return LsPaginationInterface
     */
    public function search(array $searchParams, string $locale): LsPaginationInterface
    {
        list($searchParams, $page, $limit) = $this->parseSearchParams($searchParams);

        $responseDto = $this->searchHandler->standardSearch($searchParams, $locale);
        return $this->swooxMapperChain->map($responseDto, $locale);
    }

    /**
     * @param array $searchParams
     * @param string $locale
     * @return PaginationInterface
     */
    public function paginatedSearch(array $searchParams, string $locale): PaginationInterface
    {
        list($searchParams, $page, $limit) = $this->parseSearchParams($searchParams);
        $responseDto = $this->searchHandler->standardSearch($searchParams, $locale);
        $mappedResponseDto = $this->swooxMapperChain->map($responseDto, $locale);

        return $this->paginator->paginate($mappedResponseDto, $page, $limit);
    }

    /**
     * @param array $searchParams
     * @return array
     */
    private function parseSearchParams(array $searchParams): array
    {
        if (isset($searchParams['limit'])) {
            $limit = min($this->swooxConfiguration->getDefaultLimit(), $searchParams['limit']);
        } else {
            $limit = $this->swooxConfiguration->getDefaultLimit();
        }

        if (isset($searchParams['page'])) {
            $page = max(1, $searchParams['page']);
        } else {
            $page = 1;
        }

        $searchParams['offset'] = ($page - 1) * $limit;

        unset(
            $searchParams['page'],
            $searchParams['q'],
            $searchParams['filter'],
            $searchParams['_token']
        );

        return [
            $searchParams,
            $page,
            $limit,
        ];
    }
}
