<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Sulu\SwooxMassiveSearchBundle\Adapter\Export;

use JMS\Serializer\SerializerInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\Converter\SwooxConverter;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\Event\PrePublishEvent;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\Event\PreUnpublishEvent;
use Massive\Bundle\SearchBundle\Search\AdapterInterface;
use Massive\Bundle\SearchBundle\Search\Document;
use Massive\Bundle\SearchBundle\Search\Factory;
use Massive\Bundle\SearchBundle\Search\SearchQuery;
use Massive\Bundle\SearchBundle\Search\SearchResult;
use PhpAmqpLib\Message\AMQPMessage;
use Lifestyle\DataCollector\DataCollector;
use OldSound\RabbitMqBundle\RabbitMq\ProducerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\Database\Manager as DbManger;

/**
 * Class SwooxExportAdapter
 * @package LifeStyle\Sulu\SwooxMassiveSearchBundle\Adapter\Export
 */
class SwooxExportAdapter implements AdapterInterface
{
    const CLASS_NAME = 'CmsPage';
    const TYPE = 'cms';
    const ID_PREFIX = 'cms_';
    const SEARCH_FIELD = 'CONTENT';
    const SEARCH_RESULT_DESCRIPTION = 'SEARCH_RESULT_DESCRIPTION';
    const PROPERTY_MESSAGE_DELIVERY_MODE = 'delivery_mode';
    const MESSAGE_DELIVERY_MODE_PERSISTENCE = AMQPMessage::DELIVERY_MODE_PERSISTENT;

    /**
     * @var \Massive\Bundle\SearchBundle\Search\Factory
     */
    private $factory;

    /**
     * @var array
     */
    private $responsible = [];

    /**
     * @var ProducerInterface
     */
    private $producer;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;

    /**
     * @var DbManger
     */
    private $dbManager;

    /**
     * SwooxExportAdapter constructor.
     * @param Factory $factory
     * @param ProducerInterface $producer
     * @param SerializerInterface $serializer
     * @param EventDispatcherInterface $eventDispatcher
     * @param DbManger $dbManager
     * @param array $responsible
     */
    public function __construct(
        Factory $factory,
        ProducerInterface $producer,
        SerializerInterface $serializer,
        EventDispatcherInterface $eventDispatcher,
        DbManger $dbManager,
        array $responsible
    ) {
        $this->factory = $factory;
        $this->producer = $producer;
        $this->serializer = $serializer;
        $this->eventDispatcher = $eventDispatcher;
        $this->dbManager = $dbManager;
        $this->responsible = $responsible;
    }

    /**
     * @param Document $document
     * @param string $indexName
     * @return DataCollector|null
     */
    public function index(Document $document, $indexName)
    {
        if (in_array($document->getIndex(), $this->responsible)) {
            $searchFieldCollector = '';

            $collector = new DataCollector();
            $collector->addItem('id', self::ID_PREFIX . $document->getId());
            $collector->addItem('className', self::CLASS_NAME);
            $collector->addItem('event', strtolower(self::CLASS_NAME . '.publish'));
            $collector->addItem('LOCALE', $document->getLocale());

            $collector->addItem('TYPE', self::TYPE);
            $collector->addItem('TITLE', $document->getTitle());
            $collector->addItem('URL', $document->getUrl());

            $searchFieldCollector .= ' ' . $document->getTitle();
            $searchFieldCollector .= ' ' . $document->getId();

            foreach ($document->getFields() as $field) {
                $value = $field->getValue();

                //check the status because we get also the upublished items here
                if ('state' == $field->getName() && $value !== 'published') {
                    $this->deindex($document, $indexName);
                    return null;
                }

                if (
                    is_array($value) ||
                    (
                        'published' !== $field->getName() &&
                        'authored' !== $field->getName() &&
                        (false === strpos($value, SwooxConverter::SWOOX_IDENTIFIER))
                    )
                ) {
                    continue;
                }

                //search for the search description field swoox_description_box
                if (strpos($field->getName(), 'swoox_description_box')) {
                    $cleanedValue = str_replace(SwooxConverter::SWOOX_IDENTIFIER, '', $value);
                    if (!empty($cleanedValue)) {
                        $collector->addItem(self::SEARCH_RESULT_DESCRIPTION, $cleanedValue);
                    }
                } else {
                    $collector->addItem($field->getName(), str_replace(SwooxConverter::SWOOX_IDENTIFIER, '', $value));
                }

                $searchFieldCollector .= ' ' . $value;
            }

            //add searchable field
            $collector->addItem(
                self::SEARCH_FIELD, $this->cleanUpSearchString($searchFieldCollector)
            );

            //dispatch publish event
            $event = new PrePublishEvent($collector);
            $this->eventDispatcher->dispatch(PrePublishEvent::NAME, $event);

            $this->producer->publish(
                $this->serializer->serialize($collector, 'json'),
                '',
                [self::PROPERTY_MESSAGE_DELIVERY_MODE => self::MESSAGE_DELIVERY_MODE_PERSISTENCE]
            );

            return $collector;
        }
        return null;
    }

    /**
     * @param Document $document
     * @param string $indexName
     * @return void
     */
    public function deindex(Document $document, $indexName)
    {

        if (in_array($document->getIndex(), $this->responsible)) {
            $collector = new DataCollector();
            $collector->addItem('id', self::ID_PREFIX . $document->getId());
            $collector->addItem('className', self::CLASS_NAME);
            $collector->addItem('event', strtolower(self::CLASS_NAME . '.unpublish'));
            $collector->addItem('LOCALE', $document->getLocale());
            $collector->addItem('type', self::TYPE);
            $collector->addItem('title', $document->getTitle());
            $collector->addItem('URL', $document->getUrl());

            //dispatch unpublish event
            $event = new PreUnpublishEvent($collector);
            $this->eventDispatcher->dispatch(PreUnpublishEvent::NAME, $event);

            $this->producer->publish(
                $this->serializer->serialize($collector, 'json'),
                '',
                [self::PROPERTY_MESSAGE_DELIVERY_MODE => self::MESSAGE_DELIVERY_MODE_PERSISTENCE]
            );
        }
    }

    /**
     * {@inheritdoc}
     */
    public function search(SearchQuery $searchQuery)
    {
        return new SearchResult([], 0);
    }

    /**
     * {@inheritdoc}
     */
    public function getStatus()
    {
    }

    /**
     * {@inheritdoc}
     */
    public function purge($indexName)
    {
        $cmsExportEntities = $this->dbManager->findAllCmsEntities();
        if (null !== $cmsExportEntities) {
            foreach ($cmsExportEntities as $cmsExportEntity) {
                $collector = new DataCollector();
                $collector->addItem('id', $cmsExportEntity->getId());
                $collector->addItem('className', self::CLASS_NAME);
                $collector->addItem('event', strtolower(self::CLASS_NAME . '.unpublish'));
                $collector->addItem('LOCALE', $cmsExportEntity->getLocale());
                $collector->addItem('type', self::TYPE);
                $collector->addItem('URL', $cmsExportEntity->getUrl());

                //dispatch unpublish event
                $event = new PreUnpublishEvent($collector);
                $this->eventDispatcher->dispatch(PreUnpublishEvent::NAME, $event);

                $this->producer->publish(
                    $this->serializer->serialize($collector, 'json'),
                    '',
                    [self::PROPERTY_MESSAGE_DELIVERY_MODE => self::MESSAGE_DELIVERY_MODE_PERSISTENCE]
                );
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function listIndexes()
    {
        return ['massive_swoox'];
    }

    /**
     * {@inheritdoc}
     */
    public function flush(array $indexNames)
    {
    }

    /**
     * Optimizes the index with the given name (that's zend lucene specific).
     *
     * @param string $indexName
     */
    public function optimize($indexName)
    {
    }

    /**
     * {@inheritdoc}
     */
    public function initialize()
    {
    }

    /**
     * @param mixed $value
     * @return string
     */
    private function cleanUpSearchString($value)
    {
        return trim(
            preg_replace(
                '/\s\s+/', ' ',
                str_replace(
                    ['|', '[', ']', SwooxConverter::SWOOX_IDENTIFIER], [' ', ' ', ' ', ' '],
                    strip_tags((string)$value)
                )
            )
        );
    }
}
