<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager;

use GuzzleHttp\Exception\GuzzleException;
use JMS\Serializer\SerializerInterface;
use Psr\Log\LoggerInterface;
use LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager\Model\Response\StandardResponse;
use Swoox\Api\Model\Response\Search\Filter\FilterResponse;

/**
 * Class SearchHandler
 *
 * @copyright  2018 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    LifeStyle\Sulu\SwooxMassiveSearchBundle\SearchManager
 */
class SearchHandler implements SearchHandlerInterface
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var Configuration
     */
    private $configuration;

    /**
     * SearchHandler constructor.
     * @param LoggerInterface $logger
     * @param SerializerInterface $serializer
     * @param Configuration $configuration
     */
    public function __construct(LoggerInterface $logger, SerializerInterface $serializer, Configuration $configuration)
    {
        $this->logger = $logger;
        $this->serializer = $serializer;
        $this->configuration = $configuration;
    }

    /**
     * @param array $searchParams
     * @param string $locale
     * @return null|StandardResponse
     */
    public function standardSearch(array $searchParams, $locale)
    {
        $searchParams['clientid'] = $this->configuration->getClientId();
        $searchParams['locale'] = $locale;

        if (!isset($searchParams['limit'])) {
            $searchParams['limit'] = $this->configuration->getDefaultLimit();
        }

        $searchParams[$this->configuration->getLocaleSlotId()] = $locale;
        $this->logger->debug('Swoox standard search with params ' . json_encode($searchParams));

        $url = $this->configuration->getApiDomain() . '/api/v1/search/standard.json';
        $options = ['headers' => ['Accept' => 'application/json'], 'query' => $searchParams];

        // Send swoox request
        $swooxResponse = null;
        $client = new \GuzzleHttp\Client();
        try {
            $swooxResponse = $client->request('GET', $url, $options);
        } catch (GuzzleException $exception) {
            $this->logger->error('Error while receiving products from Swoox: '. $exception->getMessage());
        }

        // Receive swoox response and parse into model
        $searchResponse = null;
        if (null !== $swooxResponse) {
            /** @var StandardResponse $response */
            try {
                $searchResponse = $this->serializer->deserialize(
                    $swooxResponse->getBody()->getContents(),
                    StandardResponse::class,
                    'json'
                );
            } catch (\JMS\Serializer\Exception\Exception $exception) {
                $this->logger->error('Error while deserializing Swoox response: '. $exception->getMessage());
            }
        }

        return $searchResponse instanceof StandardResponse ? $searchResponse : null;
    }

    /**
     * @param array $searchParams
     * @param string $locale
     * @return null|FilterResponse
     */
    public function filterSearch(array $searchParams, $locale)
    {
        $searchParams['clientid'] = $this->configuration->getClientId();
        $searchParams['locale'] = $locale;

        $searchParams[$this->configuration->getLocaleSlotId()] = $locale;

        $url = $this->configuration->getApiDomain() . '/api/v1/search/filter.json';
        $options = ['headers' => ['Accept' => 'application/json'], 'query' => $searchParams];
        $this->logger->debug('Swoox filter search with params ' . json_encode($searchParams));

        // Send swoox request
        $swooxResponse = null;
        $client = new \GuzzleHttp\Client();
        try {
            $swooxResponse = $client->request('GET', $url, $options);
        } catch (GuzzleException $exception) {
            $this->logger->error('Error while receiving data from Swoox: '. $exception->getMessage());
        }

        // Receive swoox response and parse into model
        $searchResponse = null;
        if (null !== $swooxResponse) {
            /** @var SearchResponse $response */
            try {
                $searchResponse = $this->serializer->deserialize(
                    $swooxResponse->getBody()->getContents(),
                    FilterResponse::class,
                    'json'
                );
            } catch (\JMS\Serializer\Exception\Exception $exception) {
                $this->logger->error('Error while deserializing Swoox response: '. $exception->getMessage());
            }
        }

        return $searchResponse instanceof FilterResponse ? $searchResponse : null;
    }
}
