<?php

/**
 * Class Config
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Swoox\RestApiClientBundle\Services\Item;

use Lifestyle\Swoox\RestApiClientBundle\Exception\InvalidConfigurationException;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use Symfony\Component\Validator\Validation;

/**
 * Class Config
 * @package Lifestyle\Swoox\RestApiClientBundle\Services\Item
 */
class Config
{
    /**
     * @Assert\NotBlank(message="Parameter lifestyle_swoox_rest_api_client.client_id: This value should not be blank.")
     * @Assert\GreaterThan(value="0", message="Parameter lifestyle_swoox_rest_api_client.client_id: This value should be a valid Swoox client-id.")
     *
     * @var int
     */
    private $clientId;

    /**
     * @Assert\NotBlank(message="Parameter lifestyle_swoox_rest_api_client.index_url: This value should not be blank.")
     * @Assert\Url(message="Parameter lifestyle_swoox_rest_api_client.index_url: This value is not a valid URL.")
     *
     * @var string
     */
    private $indexUrl;

    /**
     * @Assert\NotBlank()
     * @Assert\Url(message="Parameter lifestyle_swoox_rest_api_client.delete_url: This value is not a valid URL.")
     *
     * @var string
     */
    private $deleteUrl;

    /**
     * @Assert\NotBlank(message="Parameter lifestyle_swoox_rest_api_client.request_format: This value should not be blank.")
     * @Assert\Choice(choices={"json", "xml"}, message="Parameter lifestyle_swoox_rest_api_client.request_format: This value should json or xml.")
     *
     * @var string
     */
    private $requestFormat;

    /**
     * @Assert\NotBlank(message="Parameter lifestyle_swoox_rest_api_client.responsible_classes: This array should not be empty.")
     * @Assert\Type("array", message="Parameter lifestyle_swoox_rest_api_client.responsible_classes: This value should be of type {{ type }}.")
     *
     * @var array
     */
    private $responsibleClasses;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * @var bool
     */
    private $validated = false;

    /**
     * Config constructor.
     * @param int $clientId
     * @param string $indexUrl
     * @param string $deleteUrl
     * @param string $requestFormat
     * @param array $responsibleClasses
     * @param ValidatorInterface $validator
     */
    public function __construct(
        $clientId,
        $indexUrl,
        $deleteUrl,
        $requestFormat,
        $responsibleClasses,
        ValidatorInterface $validator = null
    ) {
        $this->clientId = (int)$clientId;
        $this->indexUrl = (string)$indexUrl;
        $this->deleteUrl = (string)$deleteUrl;
        $this->requestFormat = (string)$requestFormat;
        $this->responsibleClasses = (array)$responsibleClasses;
        $this->validator = $validator;
    }

    /**
     * @return int
     */
    public function getClientId()
    {
        $this->validate();
        return $this->clientId;
    }

    /**
     * @return string
     */
    public function getIndexUrl()
    {
        $this->validate();
        return $this->indexUrl;
    }

    /**
     * @return string
     */
    public function getDeleteUrl()
    {
        $this->validate();
        return $this->deleteUrl;
    }

    /**
     * @return string
     */
    public function getRequestFormat()
    {
        $this->validate();
        return $this->requestFormat;
    }

    /**
     * @return array
     */
    public function getResponsibleClasses()
    {
        $this->validate();
        return $this->responsibleClasses;
    }

    /**
     * @throws InvalidConfigurationException
     */
    private function validate()
    {
        if (!$this->validated) {
            $validator = $this->validator;
            if (null === $validator) {
                $validator = Validation::createValidatorBuilder()
                    ->enableAnnotationMapping()
                    ->getValidator();
            }
            $errors = $validator->validate($this);
            $this->validated = true;
            if (count($errors) > 0) {
                throw new InvalidConfigurationException(
                    'Invalid or missing plugin configuration for swoox export handler: ' . (string)$errors
                );
            }
        }
    }
}
