<?php

/**
 * Class Handler
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Swoox\RestApiClientBundle\Services\Item;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Swoox\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Swoox\RestApiClientBundle\Services\HandlerInterface;
use Psr\Log\LoggerInterface;

/**
 * Class Handler
 * @package Lifestyle\Swoox\RestApiClientBundle\Services\Item
 */
class Handler implements HandlerInterface
{
    const HTTP_METHOD_PUT = 'PUT';
    const HTTP_METHOD_DELETE = 'DELETE';

    /**
     * @var ClientInterface
     */
    private $client;

    /**
     * @var SerializerInterface
     */
    private $serializer;

    /**
     * @var Mapper
     */
    private $mapper;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @param ClientInterface $client
     * @param SerializerInterface $serializer
     * @param Mapper $mapper
     * @param Config $config
     * @param LoggerInterface $logger
     */
    public function __construct(
        ClientInterface $client,
        SerializerInterface $serializer,
        Mapper $mapper,
        Config $config,
        LoggerInterface $logger
    ) {
        $this->client = $client;
        $this->serializer = $serializer;
        $this->mapper = $mapper;
        $this->config = $config;
        $this->logger = $logger;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return bool
     */
    public function isResponsible(DataCollectorInterface $collector)
    {
        return in_array($collector->getItemValue('className'), $this->config->getResponsibleClasses());
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws RestRequestFailedException
     */
    public function createOrUpdate(DataCollectorInterface $collector)
    {
        $requestModel = $this->mapper->mapUpdateRequestModel($collector);
        $this->send(self::HTTP_METHOD_PUT, $this->config->getIndexUrl(), $requestModel);
    }

    /**
     * @param DataCollectorInterface $collector
     * @throws RestRequestFailedException
     */
    public function delete(DataCollectorInterface $collector)
    {
        $requestModel = $this->mapper->mapDeleteRequestModel($collector);
        $this->send(self::HTTP_METHOD_DELETE, $this->config->getDeleteUrl(), $requestModel);
    }

    /**
     * @param string $method
     * @param string $url
     * @param object $requestModel
     * @throws RestRequestFailedException
     */
    private function send($method, $url, $requestModel)
    {
        // Prepare request
        $options = array(
            'headers' => [
                'clientid' => $this->config->getClientId(),
            ],
            'body' => $this->serializer->serialize($requestModel, $this->config->getRequestFormat())
        );

        $this->logger->info(
            sprintf(
                'SwooxRequestHandler: sending request for client "%s" url %s',
                $this->config->getClientId(),
                $url
            )
        );

        $this->logger->debug('SwooxRequestHandler: request body ' . $options['body']);

        try {
            // Send request
            $webserviceResponse = $this->client->request($method, $url, $options);
        } catch (GuzzleException $exception) {
            // Received an error response
            throw new RestRequestFailedException($exception->getMessage(), $exception->getCode(), $exception);
        }

        $this->logger->info('SwooxRequestHandler: received response.');
        $this->logger->debug($webserviceResponse->getBody()->getContents());
    }
}
