<?php

/**
 * Class Mapper
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Swoox\RestApiClientBundle\Services\Item;

use Lifestyle\DataCollector\DataCollectorInterface;
use Swoox\Api\Model\Request\Delete\Swoox\Item as DeleteRequestItem;
use Swoox\Api\Model\Request\Delete\Swoox\Request as DeleteRequest;
use Swoox\Api\Model\Request\Index\Swoox\Item;
use Swoox\Api\Model\Request\Index\Swoox\NameFieldList;
use Swoox\Api\Model\Request\Index\Swoox\Request;
use Swoox\Api\Model\Request\Index\Swoox\ValueItem;

/**
 * Class Mapper
 * @package Lifestyle\Swoox\RestApiClientBundle\Services\Item
 */
class Mapper
{
    /**
     * @param DataCollectorInterface $collector
     * @return Request
     */
    public function mapUpdateRequestModel(DataCollectorInterface $collector)
    {
        $request = new Request();

        // Create one item for each localization
        if (count($collector->getItemValue('locales')) > 0) {
            foreach ($collector->getItemValue('locales') as $locale) {
                $request->addItem(
                    $this->mapItemFromDataCollector($collector, $locale)
                );
            }

            return $request;
        }

        $request->addItem(
            $this->mapItemFromDataCollector($collector)
        );

        return $request;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param string|null $locale
     * @return Item
     */
    protected function mapItemFromDataCollector(DataCollectorInterface $collector, $locale = null)
    {
        $item = (new Item())
            ->setDocumentLanguage($locale)
            ->setDocumentPrefix('')
            ->setDocumentRanking('')
            ->setDocumentUniqueKey($collector->getItemValue('id') . (null !== $locale ? '-' . $locale : ''))
            ->setDocumentTitle('')
            ->setDocumentURL('')
            ->setDocumentText('');

        if (null !== $locale) {
            $this->mapValue($item, 'LOCALE', $locale);
        }

        foreach ($collector->getItems() as $collectorItem) {
            if (null !== $locale && $collectorItem->hasLocale() && $locale !== $collectorItem->getLocale()) {
                continue;
            }

            $value = $this->prepareValue($collectorItem->getValue());

            if (is_array($value)) {
                if ('options' === $collectorItem->getKey()) {
                    $value = array_keys($value);
                }
                $this->mapMultiValues($item, $collectorItem->getKey(), $value);
            }
            if (!is_array($value)) {
                $this->mapValue($item, $collectorItem->getKey(), $value);
            }
        }

        return $item;
    }

    /**
     * @param Item $item
     * @param string $key
     * @param mixed $value
     */
    private function mapValue(Item $item, $key, $value)
    {
        $nameFieldList = new NameFieldList();
        $nameFieldList
            ->setName($key)
            ->setValue($value)
            ->setValueClean($value);

        $item->addDocumentFields($nameFieldList);
    }

    /**
     * @param Item $item
     * @param string $key
     * @param array $values
     */
    private function mapMultiValues(Item $item, $key, array $values)
    {
        $valueItems = [];
        foreach ($values as $value) {
            if (is_array($value)) {
                $value = serialize($value);
            }

            $valueItem = new ValueItem();
            $valueItem->setValue($value);
            $valueItem->setValueClean($value);
            $valueItems[] = $valueItem;
        }

        $nameFieldList = new NameFieldList();
        $nameFieldList
            ->setName($key)
            ->setValues($valueItems);

        $item->addDocumentFields($nameFieldList);
    }

    /**
     * @param mixed $value
     * @return mixed
     */
    private function prepareValue($value)
    {
        if (true === $value) {
            $value = 1;
        }
        if (false === $value) {
            $value = 0;
        }

        return $value;
    }

    /**
     * @param DataCollectorInterface $collector
     * @return DeleteRequest
     */
    public function mapDeleteRequestModel(DataCollectorInterface $collector)
    {
        $request = new DeleteRequest();

        // Create one item for each localization
        if (count($collector->getItemValue('locales')) > 0) {
            foreach ($collector->getItemValue('locales') as $locale) {
                $deleteRequestItem = new DeleteRequestItem();

                $deleteRequestItem->setDocumentUniqueKey(
                    $collector->getItemValue('id') . '-' . $locale
                );

                $request->addItem($deleteRequestItem);
            }

            return $request;
        }

        $request->addItem(
            (new DeleteRequestItem())
                ->setDocumentUniqueKey($collector->getItemValue('id'))
        );

        return $request;
    }
}
