<?php

/**
 * Class HandlerTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Swoox\RestApiClientBundle\Services\Item;

use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use GuzzleHttp\Exception\RequestException;
use JMS\Serializer\SerializerInterface;
use Lifestyle\DataCollector\DataCollector;
use Lifestyle\Swoox\RestApiClientBundle\Exception\RestRequestFailedException;
use Lifestyle\Swoox\RestApiClientBundle\Services\Item\Config;
use Lifestyle\Swoox\RestApiClientBundle\Services\Item\Handler;
use Lifestyle\Swoox\RestApiClientBundle\Services\Item\Mapper;
use Psr\Log\LoggerInterface;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Class HandlerTest
 * @package Tests\Lifestyle\Swoox\RestApiClientBundle\Services\Item
 */
class HandlerTest extends TestCase
{
    /**
     * @var Handler
     */
    private $handler;

    public function testIsResponsible()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $client = $this->getMockBuilder(ClientInterface::class)->getMock();
        $config = $this->getMockBuilder(Config::class)->disableOriginalConstructor()->getMock();
        $config->method('getResponsibleClasses')->willReturn(['ResponsibleClass']);

        $handler = new Handler($client, $serializer, $mapper, $config, $logger);

        $dataCollector = new DataCollector();

        // Swoox handler should be responsible for products.
        $this->assertFalse($handler->isResponsible($dataCollector));

        $dataCollector->addItem('className', 'NotResponsibleClass');
        $this->assertFalse($handler->isResponsible($dataCollector));

        $dataCollector->addItem('className', 'ResponsibleClass');
        $this->assertFalse($handler->isResponsible($dataCollector));
    }

    public function testPublishSuccessfully()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $validator = $this->getMockBuilder(ValidatorInterface::class)->getMock();
        $config = new Config(123456, 'unit_test_index_url', 'unit_test_delte_url', 'json', [], $validator);

        $requestOptions = [
            'headers' => [
                'clientid' => $config->getClientId(),
            ],
            'body' => null,
        ];

        $client = $this->getMockBuilder(ClientInterface::class)
            ->getMock();
        $client->expects($this->once())
            ->method('request')
            ->with(Handler::HTTP_METHOD_PUT, $config->getIndexUrl(), $requestOptions)
            ->willReturn(new Response(200, []));

        $handler = new Handler($client, $serializer, $mapper, $config, $logger);

        $dataCollector = new DataCollector();
        $handler->createOrUpdate($dataCollector);
    }

    public function testUnpublishSuccessfully()
    {
        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $validator = $this->getMockBuilder(ValidatorInterface::class)->getMock();
        $config = new Config(123456, 'unit_test_index_url', 'unit_test_delte_url', 'json', [], $validator);

        $requestOptions = [
            'headers' => [
                'clientid' => $config->getClientId(),
            ],
            'body' => null,
        ];

        $client = $this->getMockBuilder(ClientInterface::class)
            ->getMock();
        $client->expects($this->once())
            ->method('request')
            ->with(Handler::HTTP_METHOD_DELETE, $config->getDeleteUrl(), $requestOptions)
            ->willReturn(new Response(200, []));

        $handler = new Handler($client, $serializer, $mapper, $config, $logger);

        $dataCollector = new DataCollector();
        $handler->delete($dataCollector);
    }

    public function testSendFailed()
    {
        $this->expectException(RestRequestFailedException::class);
        $this->expectExceptionMessage("Unit Test Error Message");

        $serializer = $this->getMockBuilder(SerializerInterface::class)->getMock();
        $logger = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $mapper = $this->getMockBuilder(Mapper::class)->getMock();
        $validator = $this->getMockBuilder(ValidatorInterface::class)->getMock();
        $config = new Config(123456, 'unit_test_index_url', 'unit_test_delte_url', 'json', ['dummy'], $validator);

        // Create a mock and queue the responses.
        $mock = new MockHandler([
            new RequestException("Unit Test Error Message", new Request('GET', 'test')),
        ]);

        $handler = HandlerStack::create($mock);
        $client = new Client(['handler' => $handler]);

        $handler = new Handler($client, $serializer, $mapper, $config, $logger);

        $dataCollector = new DataCollector();
        $handler->delete($dataCollector);
    }
}
