<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright   2021 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

declare(strict_types=1);

namespace Lifestyle\Swoox\RestApiClientBundle\Services\Item;

use Lifestyle\DataCollector\DataCollectorInterface;
use Swoox\Api\Model\Request\Delete\Swoox\Item as DeleteRequestItem;
use Swoox\Api\Model\Request\Delete\Swoox\Request as DeleteRequest;
use Swoox\Api\Model\Request\Index\Swoox\Item;
use Swoox\Api\Model\Request\Index\Swoox\NameFieldList;
use Swoox\Api\Model\Request\Index\Swoox\Request;
use Swoox\Api\Model\Request\Index\Swoox\ValueItem;

/**
 * Class Mapper
 *
 * @package Lifestyle\Swoox\RestApiClientBundle\Services\Item
 */
class Mapper
{
    /**
     * @param DataCollectorInterface $collector
     *
     * @return Request
     */
    public function mapUpdateRequestModel(DataCollectorInterface $collector): Request
    {
        $request = new Request();

        $availableChannels = $collector->getItemValue('availableChannels');
        if (!is_array($availableChannels) || count($availableChannels) === 0) {
            $availableChannels = [null];
        }
        $locales = $collector->getItemValue('locales') ?? [null];
        if (!is_array($locales) || count($locales) === 0) {
            $locales = [null];
        }
        $assignedChannels = $collector->getItemValue('channels') ?? [];

        // Create one item for each channel and localization if available
        foreach ($availableChannels as $channel) {
            foreach ($locales as $locale) {
                $request->addItem(
                    $this->mapItemFromDataCollector(
                        $collector,
                        $locale,
                        $channel,
                        !$channel || in_array($channel, $assignedChannels)
                    )
                );
            }
        }

        return $request;
    }

    /**
     * @param DataCollectorInterface $collector
     * @param string|null            $locale
     * @param string|null            $channel
     * @param bool                   $active
     *
     * @return Item
     */
    protected function mapItemFromDataCollector(
        DataCollectorInterface $collector,
        $locale = null,
        $channel = null,
        $active = true
    ): Item {
        $uniqueKey = $collector->getItemValue('id')
            . ((null !== $locale) ? '-' . $locale : '')
            . ((null !== $channel) ? '-' . $channel : '');

        $item = (new Item())
            ->setDocumentLanguage($locale)
            ->setDocumentPrefix('')
            ->setDocumentRanking('')
            ->setDocumentUniqueKey($uniqueKey)
            ->setDocumentTitle('')
            ->setDocumentURL('')
            ->setDocumentText('');

        if (null !== $locale) {
            $this->mapValue($item, 'LOCALE', $locale);
        }

        if (null !== $channel) {
            $this->mapValue($item, 'active', $active);
            $this->mapValue($item, 'channel', $channel);
        }

        foreach ($collector->getItems() as $collectorItem) {
            // special channel based fields handling
            if (null !== $channel && false !== strpos($collectorItem->getKey(), 'swooxChannel')) {
                $key = strtolower(str_replace('swooxChannel', '', $collectorItem->getKey()));
                $this->mapValue($item, $key, $collectorItem->getValue()[$channel] ?? null);

                continue;
            }

            if (null !== $locale && $collectorItem->hasLocale() && $locale !== $collectorItem->getLocale()) {
                continue;
            }

            $value = $this->prepareValue($collectorItem->getValue());

            if (is_array($value)) {
                if ('options' === $collectorItem->getKey()) {
                    $value = array_keys($value);
                }
                $this->mapMultiValues($item, $collectorItem->getKey(), $value);
            }
            if (!is_array($value)) {
                $this->mapValue($item, $collectorItem->getKey(), $value);
            }
        }

        return $item;
    }

    /**
     * @param Item $item
     * @param string $key
     * @param mixed $value
     */
    private function mapValue(Item $item, $key, $value)
    {
        $nameFieldList = new NameFieldList();
        $nameFieldList
            ->setName($key)
            ->setValue($value)
            ->setValueClean($value);

        $item->addDocumentFields($nameFieldList);
    }

    /**
     * @param Item $item
     * @param string $key
     * @param array $values
     */
    private function mapMultiValues(Item $item, $key, array $values)
    {
        $valueItems = [];
        foreach ($values as $value) {
            if (is_array($value)) {
                $value = serialize($value);
            }

            $valueItem = new ValueItem();
            $valueItem->setValue($value);
            $valueItem->setValueClean($value);
            $valueItems[] = $valueItem;
        }

        $nameFieldList = new NameFieldList();
        $nameFieldList
            ->setName($key)
            ->setValues($valueItems);

        $item->addDocumentFields($nameFieldList);
    }

    /**
     * @param mixed $value
     * @return mixed
     */
    private function prepareValue($value)
    {
        if (true === $value) {
            $value = 1;
        }
        if (false === $value) {
            $value = 0;
        }

        return $value;
    }

    /**
     * @param DataCollectorInterface $collector
     *
     * @return DeleteRequest
     */
    public function mapDeleteRequestModel(DataCollectorInterface $collector): DeleteRequest
    {
        $request = new DeleteRequest();

        $availableChannels = $collector->getItemValue('availableChannels');
        if (!is_array($availableChannels) || count($availableChannels) === 0) {
            $availableChannels = [null];
        }
        $locales = $collector->getItemValue('locales') ?? [null];
        if (!is_array($locales) || count($locales) === 0) {
            $locales = [null];
        }

        // Create one delete item for each channel and localization if available
        foreach ($availableChannels as $channel) {
            foreach ($locales as $locale) {
                $uniqueKey = $collector->getItemValue('id')
                    . ((null !== $locale) ? '-' . $locale : '')
                    . ((null !== $channel) ? '-' . $channel : '');

                $request->addItem(
                    (new DeleteRequestItem())->setDocumentUniqueKey($uniqueKey)
                );
            }
        }

        return $request;
    }
}
