<?php

/**
 * Class MapperTest
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2017 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Tests\Lifestyle\Swoox\RestApiClientBundle\Services\Item;

use Lifestyle\DataCollector\DataCollector;
use Lifestyle\Swoox\RestApiClientBundle\Services\Item\Mapper;
use PHPUnit\Framework\TestCase;

/**
 * Class MapperTest
 * @package Tests\Lifestyle\Swoox\RestApiClientBundle\Services\Item
 */
class MapperTest extends TestCase
{
    public function testMapUpdateRequestModel()
    {
        $valId = 'unit_test_id_' . mt_rand();
        $valInt = mt_rand();
        $valString = md5(mt_rand());
        $valArray = array_fill(0, mt_rand(1, 25), md5(mt_rand()));

        $dataCollector = new DataCollector();
        $dataCollector->addItem('id', $valId);
        $dataCollector->addItem('itemInt', $valInt);
        $dataCollector->addItem('itemString', $valString);
        $dataCollector->addItem('itemArray', $valArray);
        $dataCollector->addItem('itemMultiDimensionalArray', [['sub_array1' => 'value1'], ['sub_array2' => 'value2']]);
        $dataCollector->addItem('itemTrue', true);
        $dataCollector->addItem('itemFalse', false);
        $dataCollector->addItem('options', ['size' => 'test', 'color' => 'test']);

        $stdObject = new \stdClass();
        $dataCollector->addItem('itemObject', $stdObject);

        $mapper = new Mapper();
        $swooxRequestModel = $mapper->mapUpdateRequestModel($dataCollector);

        $this->assertInstanceOf(\Swoox\Api\Model\Request\Index\Swoox\Request::class, $swooxRequestModel);
        $this->assertCount(1, $swooxRequestModel->getItems());

        $item = $swooxRequestModel->getItems()[0];

        // No additional field should be added (e. g. LOCALE)
        $this->assertCount(count($dataCollector->getItems()), $item->getDocumentFields());

        $this->assertSame($valId, $item->getDocumentUniqueKey());

        // Document fields should be in the same order as fields were added
        $index = 0;
        $this->assertSame('id', $item->getDocumentFields()[$index]->getName());
        $this->assertSame($valId, $item->getDocumentFields()[$index]->getValue());

        ++$index;
        $this->assertSame('itemInt', $item->getDocumentFields()[$index]->getName());
        $this->assertSame($valInt, $item->getDocumentFields()[$index]->getValue());

        ++$index;
        $this->assertSame('itemString', $item->getDocumentFields()[$index]->getName());
        $this->assertSame($valString, $item->getDocumentFields()[$index]->getValue());

        ++$index;
        $this->assertSame('itemArray', $item->getDocumentFields()[$index]->getName());
        $this->assertNull($item->getDocumentFields()[$index]->getValue());
        foreach ($valArray as $key => $value) {
            $this->assertSame($value, $item->getDocumentFields()[$index]->getValues()[$key]->getValue());
        }

        ++$index;
        $this->assertSame('itemMultiDimensionalArray', $item->getDocumentFields()[$index]->getName());
        $this->assertNull($item->getDocumentFields()[$index]->getValue());
        $this->assertSame('a:1:{s:10:"sub_array1";s:6:"value1";}', $item->getDocumentFields()[$index]->getValues()[0]->getValue());
        $this->assertSame('a:1:{s:10:"sub_array2";s:6:"value2";}', $item->getDocumentFields()[$index]->getValues()[1]->getValue());

        ++$index;
        $this->assertSame('itemTrue', $item->getDocumentFields()[$index]->getName());
        $this->assertSame(1, $item->getDocumentFields()[$index]->getValue());

        ++$index;
        $this->assertSame('itemFalse', $item->getDocumentFields()[$index]->getName());
        $this->assertSame(0, $item->getDocumentFields()[$index]->getValue());

        ++$index;
        $this->assertSame('options', $item->getDocumentFields()[$index]->getName());
        $this->assertNull($item->getDocumentFields()[$index]->getValue());
        $this->assertSame('size', $item->getDocumentFields()[$index]->getValues()[0]->getValue());
        $this->assertSame('color', $item->getDocumentFields()[$index]->getValues()[1]->getValue());

        ++$index;
        $this->assertSame('itemObject', $item->getDocumentFields()[$index]->getName());
        $this->assertSame($stdObject, $item->getDocumentFields()[$index]->getValue());
    }

    public function testMapUpdateRequestModelMultiLanguage()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('id', 'unit_test_id');
        $dataCollector->addItem('locales', ['de_DE', 'en_US']);

        $dataCollector->addItem('itemInt', 12345);
        $dataCollector->addItem('itemString', 'unit_test_string_de_DE', 'de_DE');
        $dataCollector->addItem('itemString', 'unit_test_string_en_US', 'en_US');

        $mapper = new Mapper();
        $swooxRequestModel = $mapper->mapUpdateRequestModel($dataCollector);

        $this->assertInstanceOf(\Swoox\Api\Model\Request\Index\Swoox\Request::class, $swooxRequestModel);
        $this->assertCount(2, $swooxRequestModel->getItems());

        foreach ($dataCollector->getItemValue('locales') as $key => $locale) {
            $this->assertSame('unit_test_id-' . $locale, $swooxRequestModel->getItems()[$key]->getDocumentUniqueKey());

            $this->assertSame('LOCALE', $swooxRequestModel->getItems()[$key]->getDocumentFields()[0]->getName());
            $this->assertSame($locale, $swooxRequestModel->getItems()[$key]->getDocumentFields()[0]->getValue());

            $this->assertSame('id', $swooxRequestModel->getItems()[$key]->getDocumentFields()[1]->getName());
            $this->assertSame('unit_test_id', $swooxRequestModel->getItems()[$key]->getDocumentFields()[1]->getValue());

            $this->assertSame('locales', $swooxRequestModel->getItems()[$key]->getDocumentFields()[2]->getName());
            $this->assertSame('de_DE', $swooxRequestModel->getItems()[$key]->getDocumentFields()[2]->getValues()[0]->getValue());
            $this->assertSame('en_US', $swooxRequestModel->getItems()[$key]->getDocumentFields()[2]->getValues()[1]->getValue());

            $this->assertSame('itemInt', $swooxRequestModel->getItems()[$key]->getDocumentFields()[3]->getName());
            $this->assertSame(12345, $swooxRequestModel->getItems()[$key]->getDocumentFields()[3]->getValue());

            $this->assertSame('itemString', $swooxRequestModel->getItems()[$key]->getDocumentFields()[4]->getName());
            $this->assertSame('unit_test_string_' . $locale, $swooxRequestModel->getItems()[$key]->getDocumentFields()[4]->getValue());
        }
    }

    public function testMapDeleteRequestModel()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('id', 'unit_test_id_delete');

        $mapper = new Mapper();
        $swooxRequestModel = $mapper->mapDeleteRequestModel($dataCollector);

        $this->assertInstanceOf(\Swoox\Api\Model\Request\Delete\Swoox\Request::class, $swooxRequestModel);

        $this->assertCount(1, $swooxRequestModel->getItems());
        $this->assertSame('unit_test_id_delete', $swooxRequestModel->getItems()[0]->getDocumentUniqueKey());
    }

    public function testMapDeleteRequestModelMultiLanguage()
    {
        $dataCollector = new DataCollector();
        $dataCollector->addItem('id', 'unit_test_id_delete');
        $dataCollector->addItem('locales', ['de_DE', 'en_US']);

        $mapper = new Mapper();
        $swooxRequestModel = $mapper->mapDeleteRequestModel($dataCollector);

        $this->assertInstanceOf(\Swoox\Api\Model\Request\Delete\Swoox\Request::class, $swooxRequestModel);
        $this->assertCount(2, $swooxRequestModel->getItems());

        foreach ($dataCollector->getItemValue('locales') as $key => $locale) {
            $this->assertSame('unit_test_id_delete-' . $locale, $swooxRequestModel->getItems()[$key]->getDocumentUniqueKey());
        }
    }
}
