<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Account\Consumer\Services\Address\Version1;

use Doctrine\ORM\EntityManager;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\Account\Consumer\Entity\Address;
use Lifestyle\Sylius\Account\Consumer\Services\Shared\Helper\DatabaseTrait;
use Lifestyle\Sylius\RestApiClientBundle\Services\HandlerInterface;
use Psr\Log\LoggerInterface;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\AddressRepository;
use Sylius\Component\Resource\Factory\FactoryInterface as ResourceFactory;

/**
 * Class AddressServiceHandler
 *
 * @package Lifestyle\Sylius\Account\Consumer\Services\Address\Version1
 */
class AddressServiceHandler implements HandlerInterface
{
    const HANDLER_ID = 'Address';

    use DatabaseTrait;

    /**
     * @var ResourceFactory
     */
    private $addressFactory;

    /**
     * @var AddressRepository
     */
    private $addressRepository;

    /**
     * @var EntityManager
     */
    private $addressManager;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * AddressServiceHandler constructor.
     *
     * @param ResourceFactory   $addressFactory
     * @param AddressRepository $addressRepository
     * @param EntityManager     $addressManager
     * @param LoggerInterface   $logger
     */
    public function __construct(
        ResourceFactory $addressFactory,
        AddressRepository $addressRepository,
        EntityManager $addressManager,
        LoggerInterface $logger
    ) {
        $this->addressFactory = $addressFactory;
        $this->addressRepository = $addressRepository;
        $this->addressManager = $addressManager;
        $this->logger = $logger;
    }

    /**
     * @inheritDoc
     */
    public function isResponsible(DataCollectorInterface $collector)
    {
        return $collector->getItemValue('className') === self::HANDLER_ID;
    }

    /**
     * @inheritDoc
     */
    public function createOrUpdate(DataCollectorInterface $collector)
    {
        try {
            // reconnect database if connection was lost
            $this->addressManager = $this->getClearedEntityManager($this->addressManager);

            /** @var Address $addressEntity */
            $addressEntity = $this->addressRepository->findOneBy(['code' => $collector->getItemValue('id')]);

            if (null === $addressEntity) {
                $addressEntity = $this->addressFactory->createNew();
                $addressEntity->setCode($collector->getItemValue('id'));
            }

            $addressEntity->setFirstName($collector->getItemValue('firstName'));
            $addressEntity->setLastName($collector->getItemValue('lastName'));
            $addressEntity->setCompany($collector->getItemValue('companyName'));
            $addressEntity->setStreet($collector->getItemValue('street'));
            $addressEntity->setCity($collector->getItemValue('city'));
            $addressEntity->setPostcode($collector->getItemValue('zipCode'));
            $addressEntity->setCountryCode($collector->getItemValue('countryCode'));
            $addressEntity->setPhoneNumber($collector->getItemValue('phoneNumber'));

            $this->addressRepository->add($addressEntity);
            $this->logger->info('Saved address with code: ' . $collector->getItemValue('id'));
        } catch (\Exception $exception) {
            $this->logger->error($exception->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function delete(DataCollectorInterface $collector)
    {
        // addresses will currently not be deleted
    }
}
