<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Account\Consumer\Services\Account\Version1;

use Doctrine\ORM\EntityManager;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\Account\Consumer\Entity\Customer;
use Lifestyle\Sylius\Account\Consumer\Entity\Address;
use Lifestyle\Sylius\Account\Consumer\Services\Shared\Helper\DatabaseTrait;
use Lifestyle\Sylius\RestApiClientBundle\Services\HandlerInterface;
use Psr\Log\LoggerInterface;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\AddressRepository;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\CustomerRepository;
use Sylius\Bundle\ResourceBundle\Doctrine\ORM\EntityRepository;
use Sylius\Component\Resource\Factory\FactoryInterface as ResourceFactory;
use Sylius\Component\User\Model\UserInterface;

/**
 * Class AccountServiceHandler
 *
 * @package Lifestyle\Sylius\Account\Consumer\Services\Account\Version1
 */
class AccountServiceHandler implements HandlerInterface
{
    const HANDLER_ID = 'Account';

    use DatabaseTrait;

    /**
     * @var ResourceFactory
     */
    private $customerFactory;

    /**
     * @var CustomerRepository
     */
    private $customerRepository;

    /**
     * @var EntityManager
     */
    private $customerManager;

    /**
     * @var AddressRepository
     */
    private $addressRepository;

    /**
     * @var EntityRepository
     */
    private $customerGroupRepository;

    /**
     * @var ResourceFactory
     */
    private $shopUserFactory;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Handler constructor.
     *
     * @param ResourceFactory    $customerFactory
     * @param CustomerRepository $customerRepository
     * @param EntityManager      $customerManager
     * @param AddressRepository  $addressRepository
     * @param EntityRepository   $customerGroupRepository
     * @param ResourceFactory    $shopUserFactory
     * @param LoggerInterface    $logger
     */
    public function __construct(
        ResourceFactory $customerFactory,
        CustomerRepository $customerRepository,
        EntityManager $customerManager,
        AddressRepository $addressRepository,
        EntityRepository $customerGroupRepository,
        ResourceFactory $shopUserFactory,
        LoggerInterface $logger
    ) {
        $this->customerFactory = $customerFactory;
        $this->customerRepository = $customerRepository;
        $this->customerManager = $customerManager;
        $this->addressRepository = $addressRepository;
        $this->customerGroupRepository = $customerGroupRepository;
        $this->shopUserFactory = $shopUserFactory;
        $this->logger = $logger;
    }

    /**
     * @inheritDoc
     */
    public function isResponsible(DataCollectorInterface $collector)
    {
        return $collector->getItemValue('className') === self::HANDLER_ID;
    }

    /**
     * @inheritDoc
     */
    public function createOrUpdate(DataCollectorInterface $collector)
    {
        try {
            // reconnect database if connection was lost
            $this->customerManager = $this->getClearedEntityManager($this->customerManager);

            // find customer
            /** @var Customer $customerEntity */
            $customerEntity = $this->customerRepository->findOneBy(['code' => $collector->getItemValue('id')]);
            // create if not found
            if (null === $customerEntity) {
                $customerEntity = $this->customerFactory->createNew();
                $customerEntity->setCode($collector->getItemValue('id'));
            }

            // set customer values
            $customerEntity->setFirstName($collector->getItemValue('firstName'));
            $customerEntity->setLastName($collector->getItemValue('lastName'));
            $customerEntity->setEmail($collector->getItemValue('email'));
            $customerEntity->setPhoneNumber($collector->getItemValue('phoneNumber'));
            $customerEntity->setCompanyName($collector->getItemValue('companyName'));
            $customerEntity->setTaxId($collector->getItemValue('taxId'));
            $customerEntity->setGender($collector->getItemValue('gender'));
            // set customergroup entity
            $customerEntity->setGroup(
                $this->customerGroupRepository->findOneBy(['code' => $collector->getItemValue('CustomerGroup')])
            );

            $billingAddressCodes = $collector->getItemValue('BillingAddressIds')[0];
            $shippingAddressCodes = $collector->getItemValue('ShippingAddressIds');
            $savedAddressCodes = array_unique(array_merge([$billingAddressCodes], $shippingAddressCodes));
            $addAddressCodes = $savedAddressCodes;
            $defaultAddress = null;

            // unset default address, otherwise addresses can't be deleted completely
            $customerEntity->setDefaultAddress($defaultAddress);
            // persist customer for address->customer relation
            $this->customerRepository->add($customerEntity);

            // remove addresses from customer that were removed
            /** @var Address $customerShippingaddress */
            foreach ($customerEntity->getAddresses() as $customerAddress) {
                if (null === $defaultAddress && $customerAddress->getCode() === $billingAddressCodes) {
                    $defaultAddress = $customerAddress;
                }

                if (in_array($customerAddress->getCode(), $savedAddressCodes)) {
                    $addAddressCodes = array_diff($addAddressCodes, [$customerAddress->getCode()]);
                    continue;
                }

                $customerEntity->removeAddress($customerAddress);
                $this->addressRepository->remove($customerAddress);

                $this->logger->info('Address (' . $customerAddress->getCode() .  ') for Account (' . $customerEntity->getCode() . ') deleted');
            }

            // add new addresses to customer
            foreach ($addAddressCodes as $code) {
                $addressMaster = $this->addressRepository->findOneBy([
                    'code' => $code,
                    'masterAddress' => 1
                ]);
                $address = clone $addressMaster;

                if (null !== $address) {
                    $address->setMasterAddress(0);
                    $this->addressRepository->add($address);
                    $customerEntity->addAddress($address);

                    if (null === $defaultAddress && $code === $billingAddressCodes) {
                        $defaultAddress = $address;
                    }

                    $this->logger->info('Address (' . $address->getCode() .  ') for Account (' . $customerEntity->getCode() . ') saved');
                }
            }

            // map default address
            $customerEntity->setDefaultAddress($defaultAddress);

            // find shop user
            /** @var UserInterface $shopUser */
            $shopUser = $customerEntity->getUser();
            // create if not found
            if (null === $shopUser) {
                $shopUser = $this->shopUserFactory->createNew();
                // just set a random password here. the customer should set his password with forgot password functionality
                $shopUser->setPlainPassword(substr(str_shuffle('0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'), 1, 10));
                $shopUser->setEnabled(true);
                $shopUser->setLocked(false);
                $shopUser->setCreatedAt(new \DateTime());
                $shopUser->addRole(UserInterface::DEFAULT_ROLE);
                $customerEntity->setUser($shopUser);

                $this->logger->info('Shop User for Account (' . $customerEntity->getCode() . ') created');
            }

            $shopUser->setUsername($collector->getItemValue('email'));
            $shopUser->setUsernameCanonical($collector->getItemValue('email'));
            $shopUser->setUpdatedAt(new \DateTime());

            $this->customerRepository->add($customerEntity);

            $this->logger->info('Shop User for Account (' . $customerEntity->getCode() . ') saved');
        } catch (\Exception $exception) {
            $this->logger->error($exception->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function delete(DataCollectorInterface $collector)
    {
        try {
            // reconnect database if connection was lost
            $this->customerManager = $this->getClearedEntityManager($this->customerManager);

            /** @var Customer $customerEntity */
            $customerEntity = $this->customerRepository->findOneBy(['code' => $collector->getItemValue('id')]);
            if (null !== $customerEntity && null !== $customerEntity->getUser()) {
                $customerEntity->getUser()->setEnabled(false);
                $this->customerRepository->add($customerEntity);
                $this->logger->info('Account with code: ' . $collector->getItemValue('id') . ' disabled');
            }
        } catch (\Exception $exception) {
            $this->logger->error($exception->getMessage());
        }
    }
}
