<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Account\Consumer\Services\Location\Version1;

use Doctrine\ORM\EntityManager;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\Account\Consumer\Entity\Location;
use Lifestyle\Sylius\Account\Consumer\Exception\LocationInvalidException;
use Lifestyle\Sylius\Account\Consumer\Services\Shared\Helper\DatabaseTrait;
use Lifestyle\Sylius\RestApiClientBundle\Services\HandlerInterface;
use Psr\Log\LoggerInterface;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\AddressRepository;
use Sylius\Bundle\ResourceBundle\Doctrine\ORM\EntityRepository;
use Sylius\Component\Resource\Factory\FactoryInterface as ResourceFactory;
use Symfony\Component\Validator\Validator\ValidatorInterface;

/**
 * Class LocationServiceHandler
 *
 * @package Lifestyle\Sylius\Account\Consumer\Services\Location\Version1
 */
class LocationServiceHandler implements HandlerInterface
{
    const HANDLER_ID = 'Location';

    use DatabaseTrait;

    /**
     * @var ResourceFactory
     */
    private $locationFactory;

    /**
     * @var EntityRepository
     */
    private $locationRepository;

    /**
     * @var EntityManager
     */
    private $locationManager;

    /**
     * @var AddressRepository
     */
    private $addressRepository;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var ValidatorInterface
     */
    private $validator;

    /**
     * LocationServiceHandler constructor.
     *
     * @param ResourceFactory    $locationFactory
     * @param EntityRepository   $locationRepository
     * @param EntityManager      $locationManager
     * @param AddressRepository  $addressRepository
     * @param ValidatorInterface $validator
     * @param LoggerInterface    $logger
     */
    public function __construct(
        ResourceFactory $locationFactory,
        EntityRepository $locationRepository,
        EntityManager $locationManager,
        AddressRepository $addressRepository,
        ValidatorInterface $validator,
        LoggerInterface $logger
    ) {
        $this->locationFactory = $locationFactory;
        $this->locationRepository = $locationRepository;
        $this->locationManager = $locationManager;
        $this->addressRepository = $addressRepository;
        $this->validator = $validator;
        $this->logger = $logger;
    }

    /**
     * @inheritDoc
     */
    public function isResponsible(DataCollectorInterface $collector)
    {
        return $collector->getItemValue('className') === self::HANDLER_ID;
    }

    /**
     * @param DataCollectorInterface $collector
     *
     * @throws LocationInvalidException
     */
    public function createOrUpdate(DataCollectorInterface $collector)
    {
        // reconnect database if connection was lost
        $this->locationManager = $this->getClearedEntityManager($this->locationManager);

        /** @var Location $location */
        $location = $this->locationRepository->findOneBy(['code' => $collector->getItemValue('id')]);

        if (null === $location) {
            $location = $this->locationFactory->createNew();
            $location->setCode($collector->getItemValue('id'));
        }

        $location->setName($collector->getItemValue('name'));

        $billingAddress = $this->addressRepository->findOneBy(['code' => $collector->getItemValue('BillingAddress')]);
        if (null !== $billingAddress) {
            $location->setBillingAddress($billingAddress);
        }
        $shippingAddress = $this->addressRepository->findOneBy(['code' => $collector->getItemValue('ShippingAddress')]);
        if (null !== $shippingAddress) {
            $location->setShippingAddress($shippingAddress);
        }

        // handle geopoint information
        if ($collector->hasItem('geoLocation')) {
            $location->setGeopointLatitude($collector->getItemValue('geoLocation')['Latitude']);
            $location->setGeopointLongitude($collector->getItemValue('geoLocation')['Longitude']);
        }

        $this->handleAdditionalData($collector, $location);

        $errors = $this->validator->validate($location);

        if ($errors->count() > 0) {
            throw new LocationInvalidException('Invalid Location Data: ' . (string) $errors);
        }

        $this->locationRepository->add($location);
        $this->logger->info('Successfully stored location with code: ' . $collector->getItemValue('id'));
    }

    /**
     * @inheritDoc
     */
    public function delete(DataCollectorInterface $collector)
    {
        // locations will currently not be deleted
    }

    /**
     * @param DataCollectorInterface $collector
     * @param Location               $location
     */
    protected function handleAdditionalData(DataCollectorInterface $collector, Location $location)
    {
        // do nothing in base service. extend class and overwrite method for additional field handling.
    }
}
