<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Account\Consumer\Tests\Services\Address\Version1;

use Doctrine\DBAL\Connection;
use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\Configuration;
use Doctrine\ORM\EntityManager;
use Doctrine\ORM\ORMInvalidArgumentException;
use Doctrine\ORM\Query\Expr;
use Doctrine\ORM\QueryBuilder;
use Lifestyle\DataCollector\DataCollectorInterface;
use LifeStyle\Sylius\DataModelPlugin\Entity\Address;
use Lifestyle\Sylius\Account\Consumer\Services\Address\Version1\AddressServiceHandler;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;
use Sylius\Bundle\CoreBundle\Doctrine\ORM\AddressRepository;
use Sylius\Component\Resource\Factory\FactoryInterface as ResourceFactory;

/**
 * Class AddressServiceHandlerTest
 *
 * @package Lifestyle\Sylius\Account\Consumer\Tests\Services\Address\Version1
 */
class AddressServiceHandlerTest extends TestCase
{
    public function testIsResponsibleTrue()
    {
        $dataCollectorMock = $this->getMockBuilder(DataCollectorInterface::class)->disableOriginalConstructor()->getMock();
        $dataCollectorMock->expects($this->once())->method('getItemValue')->with($this->equalTo('className'))->willReturn('Address');

        $addressFactoryMock = $this->getMockBuilder(ResourceFactory::class)->disableOriginalConstructor()->getMock();
        $addressRepositoryMock = $this->getMockBuilder(AddressRepository::class)->disableOriginalConstructor()->getMock();
        $addressManagerMock = $this->getMockBuilder(EntityManager::class)->disableOriginalConstructor()->getMock();
        $loggerMock = $this->getMockBuilder(LoggerInterface::class)->disableOriginalConstructor()->getMock();
        $testObj = new AddressServiceHandler($addressFactoryMock, $addressRepositoryMock, $addressManagerMock, $loggerMock);

        $this->assertTrue($testObj->isResponsible($dataCollectorMock));
    }

    public function testIsResponsibleFalse()
    {
        $dataCollectorMock = $this->getMockBuilder(DataCollectorInterface::class)->disableOriginalConstructor()->getMock();
        $dataCollectorMock->expects($this->once())->method('getItemValue')->with($this->equalTo('className'))->willReturn('InvalidClass');

        $addressFactoryMock = $this->getMockBuilder(ResourceFactory::class)->disableOriginalConstructor()->getMock();
        $addressRepositoryMock = $this->getMockBuilder(AddressRepository::class)->disableOriginalConstructor()->getMock();
        $addressManagerMock = $this->getMockBuilder(EntityManager::class)->disableOriginalConstructor()->getMock();
        $loggerMock = $this->getMockBuilder(LoggerInterface::class)->disableOriginalConstructor()->getMock();
        $testObj = new AddressServiceHandler($addressFactoryMock, $addressRepositoryMock, $addressManagerMock, $loggerMock);

        $this->assertFalse($testObj->isResponsible($dataCollectorMock));
    }

    public function testCreateOrUpdateCreate()
    {
        $code = 456;
        $firstName = 'testFN';
        $lastName = 'testLN';
        $companyName = null;
        $street = 'testStreet';
        $city = 'testCity';
        $zipCode = '12345';
        $countryCode = 'DE';
        $phoneNumber = null;

        $dataCollectorMock = $this->getMockBuilder(DataCollectorInterface::class)->disableOriginalConstructor()->getMock();
        $dataCollectorMock->expects($this->exactly(11))->method('getItemValue')
            ->withConsecutive(
                [$this->equalTo('id')],
                [$this->equalTo('id')],
                [$this->equalTo('firstName')],
                [$this->equalTo('lastName')],
                [$this->equalTo('companyName')],
                [$this->equalTo('street')],
                [$this->equalTo('city')],
                [$this->equalTo('zipCode')],
                [$this->equalTo('countryCode')],
                [$this->equalTo('phoneNumber')],
                [$this->equalTo('id')]
            )
            ->willReturnOnConsecutiveCalls(
                $code,
                $code,
                $firstName,
                $lastName,
                $companyName,
                $street,
                $city,
                $zipCode,
                $countryCode,
                $phoneNumber,
                $code
            );

        $newAddressEntityMock = $this->getMockBuilder(Address::class)->disableOriginalConstructor()->getMock();
        $newAddressEntityMock->expects($this->once())->method('setCode')->with($this->equalTo($code));
        $newAddressEntityMock->expects($this->once())->method('setFirstName')->with($this->equalTo($firstName));

        $addressFactoryMock = $this->getMockBuilder(ResourceFactory::class)->disableOriginalConstructor()->getMock();
        $addressFactoryMock->expects($this->once())->method('createNew')->willReturn($newAddressEntityMock);

        $queryBuilderExpressionMock = $this->getMockBuilder(Expr::class)->disableOriginalConstructor()->getMock();
        $queryMock = $this->getMockBuilder(AbstractQuery::class)->disableOriginalConstructor()->getMock();
        $queryMock->expects($this->once())->method('getResult')->willReturn([]);

        $queryBuilderMock = $this->getMockBuilder(QueryBuilder::class)->disableOriginalConstructor()->getMock();
        $queryBuilderMock->expects($this->any())->method($this->logicalAnd($this->logicalNot($this->equalTo('expr')), $this->logicalNot($this->equalTo('getQuery'))))->willReturnSelf();
        $queryBuilderMock->expects($this->any())->method('expr')->willReturn($queryBuilderExpressionMock);
        $queryBuilderMock->expects($this->any())->method('getQuery')->willReturn($queryMock);

        $addressRepositoryMock = $this->getMockBuilder(AddressRepository::class)->disableOriginalConstructor()->getMock();
        $addressRepositoryMock->expects($this->once())->method('add')->with($this->equalTo($newAddressEntityMock));
        $addressRepositoryMock->expects($this->once())->method('createQueryBuilder')->willReturn($queryBuilderMock);

        // mock database reconnect stuff
        $dbalConnectionMock = $this->getMockBuilder(Connection::class)->disableOriginalConstructor()->getMock();
        $dbalConfigurationMock = $this->getMockBuilder(Configuration::class)->disableOriginalConstructor()->getMock();
        $addressManagerMock = $this->getMockBuilder(EntityManager::class)->disableOriginalConstructor()->getMock();
        $addressManagerMock->expects($this->any())->method('getConnection')->willReturn($dbalConnectionMock);
        $addressManagerMock->expects($this->any())->method('getConfiguration')->willReturn($dbalConfigurationMock);
        $addressManagerMock->expects($this->any())->method('isOpen')->willReturn(true);

        $loggerMock = $this->getMockBuilder(LoggerInterface::class)->disableOriginalConstructor()->getMock();
        $testObj = new AddressServiceHandler($addressFactoryMock, $addressRepositoryMock, $addressManagerMock, $loggerMock);
        $testObj->createOrUpdate($dataCollectorMock);
    }

    public function testCreateOrUpdateUpdate()
    {
        $code = 456;
        $firstName = 'testFN';
        $lastName = 'testLN';
        $companyName = null;
        $street = 'testStreet';
        $city = 'testCity';
        $zipCode = '12345';
        $countryCode = 'DE';
        $phoneNumber = null;

        $dataCollectorMock = $this->getMockBuilder(DataCollectorInterface::class)->disableOriginalConstructor()->getMock();
        $dataCollectorMock->expects($this->exactly(10))->method('getItemValue')
            ->withConsecutive(
                [$this->equalTo('id')],
                [$this->equalTo('firstName')],
                [$this->equalTo('lastName')],
                [$this->equalTo('companyName')],
                [$this->equalTo('street')],
                [$this->equalTo('city')],
                [$this->equalTo('zipCode')],
                [$this->equalTo('countryCode')],
                [$this->equalTo('phoneNumber')],
                [$this->equalTo('id')]
            )
            ->willReturnOnConsecutiveCalls(
                $code,
                $firstName,
                $lastName,
                $companyName,
                $street,
                $city,
                $zipCode,
                $countryCode,
                $phoneNumber,
                $code
            );

        $addressEntityMock = $this->getMockBuilder(Address::class)->disableOriginalConstructor()->getMock();
        $addressEntityMock->expects($this->never())->method('setCode');
        $addressEntityMock->expects($this->once())->method('setFirstName')->with($this->equalTo($firstName));

        $addressFactoryMock = $this->getMockBuilder(ResourceFactory::class)->disableOriginalConstructor()->getMock();
        $addressFactoryMock->expects($this->never())->method('createNew');

        $queryBuilderExpressionMock = $this->getMockBuilder(Expr::class)->disableOriginalConstructor()->getMock();
        $queryMock = $this->getMockBuilder(AbstractQuery::class)->disableOriginalConstructor()->getMock();
        $queryMock->expects($this->once())->method('getResult')->willReturn([$addressEntityMock]);

        $queryBuilderMock = $this->getMockBuilder(QueryBuilder::class)->disableOriginalConstructor()->getMock();
        $queryBuilderMock->expects($this->any())->method($this->logicalAnd($this->logicalNot($this->equalTo('expr')), $this->logicalNot($this->equalTo('getQuery'))))->willReturnSelf();
        $queryBuilderMock->expects($this->any())->method('expr')->willReturn($queryBuilderExpressionMock);
        $queryBuilderMock->expects($this->any())->method('getQuery')->willReturn($queryMock);

        $addressRepositoryMock = $this->getMockBuilder(AddressRepository::class)->disableOriginalConstructor()->getMock();
        $addressRepositoryMock->expects($this->once())->method('add')->with($this->equalTo($addressEntityMock));
        $addressRepositoryMock->expects($this->once())->method('createQueryBuilder')->willReturn($queryBuilderMock);

        // mock database reconnect stuff
        $dbalConnectionMock = $this->getMockBuilder(Connection::class)->disableOriginalConstructor()->getMock();
        $dbalConfigurationMock = $this->getMockBuilder(Configuration::class)->disableOriginalConstructor()->getMock();
        $addressManagerMock = $this->getMockBuilder(EntityManager::class)->disableOriginalConstructor()->getMock();
        $addressManagerMock->expects($this->any())->method('getConnection')->willReturn($dbalConnectionMock);
        $addressManagerMock->expects($this->any())->method('getConfiguration')->willReturn($dbalConfigurationMock);
        $addressManagerMock->expects($this->any())->method('isOpen')->willReturn(true);

        $loggerMock = $this->getMockBuilder(LoggerInterface::class)->disableOriginalConstructor()->getMock();
        $testObj = new AddressServiceHandler($addressFactoryMock, $addressRepositoryMock, $addressManagerMock, $loggerMock);
        $testObj->createOrUpdate($dataCollectorMock);
    }

    public function testCreateOrUpdateException()
    {
        $code = 456;
        $errorMessage = 'someTestError';

        $dataCollectorMock = $this->getMockBuilder(DataCollectorInterface::class)->disableOriginalConstructor()->getMock();
        $dataCollectorMock->expects($this->never())->method('getItemValue');

        $addressFactoryMock = $this->getMockBuilder(ResourceFactory::class)->disableOriginalConstructor()->getMock();
        $addressFactoryMock->expects($this->never())->method('createNew');

        $addressRepositoryMock = $this->getMockBuilder(AddressRepository::class)->disableOriginalConstructor()->getMock();
        $addressRepositoryMock->expects($this->never())->method('add');

        // mock database reconnect stuff
        $dbalConnectionMock = $this->getMockBuilder(Connection::class)->disableOriginalConstructor()->getMock();
        $dbalConfigurationMock = $this->getMockBuilder(Configuration::class)->disableOriginalConstructor()->getMock();
        $addressManagerMock = $this->getMockBuilder(EntityManager::class)->disableOriginalConstructor()->getMock();
        $addressManagerMock->expects($this->any())->method('getConnection')->willReturn($dbalConnectionMock);
        $addressManagerMock->expects($this->any())->method('getConfiguration')->willReturn($dbalConfigurationMock);
        $addressManagerMock->expects($this->any())->method('isOpen')->willReturn(true);
        $addressManagerMock->expects($this->any())->method('clear')->willThrowException(new ORMInvalidArgumentException($errorMessage));

        $loggerMock = $this->getMockBuilder(LoggerInterface::class)->disableOriginalConstructor()->getMock();
        $loggerMock->expects($this->once())->method('error')->with($this->equalTo($errorMessage));
        $testObj = new AddressServiceHandler($addressFactoryMock, $addressRepositoryMock, $addressManagerMock, $loggerMock);
        $testObj->createOrUpdate($dataCollectorMock);
    }
}
