<?php

/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is private by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 */

namespace Lifestyle\Sylius\Asset\AssetHelper;

use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityRepository;
use Lifestyle\DataCollector\DataCollectorInterface;
use Lifestyle\Sylius\Asset\Entity\AssetStorage;
use Lifestyle\Sylius\Asset\Entity\AssetStorageMetaData;
use Lifestyle\Sylius\Asset\Exception\DatabaseException;
use Lifestyle\Sylius\Asset\Exception\InvalidArgumentException;

/**
 * Class AssetsHelper
 *
 * @copyright  2019 Lifestyle Webconsulting GmbH
 * @link       http://www.life-style.de
 * @package    Lifestyle\Sylius\Product\ProductHelper
 */
class AssetHelper
{
    use DatabaseTrait;

    /**
     * @var EntityRepository
     */
    private $assetRepository;

    /**
     * @var EntityManager
     */
    private $assetManager;

    /**
     * AssetsHelper constructor.
     * @param EntityRepository $assetRepository
     * @param EntityManager $assetManager
     */
    public function __construct(EntityRepository $assetRepository, EntityManager $assetManager)
    {
        $this->assetRepository = $assetRepository;
        $this->assetManager = $assetManager;
    }

    /**
     * @param string $resourceId
     * @throws \Doctrine\Common\Persistence\Mapping\MappingException
     * @throws \Doctrine\ORM\ORMException
     * @throws DatabaseException
     */
    public function updateAsset(DataCollectorInterface $dataCollector)
    {
        $this->assetManager = $this->getClearedEntityManager($this->assetManager);

        $resourceId = (string)$dataCollector->getItemValue('id');
        $path = (string)$dataCollector->getItemValue('path');
        $filename = (string)$dataCollector->getItemValue('filename');

        if (0 === strlen($resourceId)) {
            throw new InvalidArgumentException(
                'Cannot proceed asset-update! Asset-Id not set in message.'
            );
        }

        $assetsEntity = $this->assetRepository->findOneBy(['resourceId' => $resourceId]);
        if (null === $assetsEntity) {
            $assetsEntity = $this->getNewAssetsStorage();
            $assetsEntity->setResourceId($resourceId);
        } else {
            //remove all meta
            if (null !== $assetsEntity->getMetaDatas()) {
                foreach ($assetsEntity->getMetaDatas() as $metaData) {
                    $this->assetManager->remove($metaData);
                }
                $this->flush($this->assetManager);
            }
        }

        $assetsEntity->setPath($path);
        $assetsEntity->setName($filename);

        $this->assetManager->persist($assetsEntity);
        $this->flush($this->assetManager);

        foreach ($dataCollector->getItems() as $collectorItem) {
            $key = $collectorItem->getKey();
            if (strpos($key, 'metadata_') === 0) {
                $metaData = $this->getNewAssetsStorageMetaData();
                $metaData->setValue((string)$collectorItem->getValue());
                $metaData->setLocale((string)$collectorItem->getLocale());
                $metaData->setType(substr($key, 9));
                $metaData->setAssetStorage($assetsEntity);
                $assetsEntity->addMetaData($metaData);
            }
        }

        $this->assetManager->persist($assetsEntity);
        $this->flush($this->assetManager);
    }

    /**
     * @param string $resourceId
     * @throws \Doctrine\ORM\ORMException
     * @throws DatabaseException
     * @throws \Doctrine\Common\Persistence\Mapping\MappingException
     */
    public function removeAsset(string $resourceId)
    {
        $this->assetManager = $this->getClearedEntityManager($this->assetManager);

        $resourceId = (string)$resourceId;
        if (0 === strlen($resourceId)) {
            throw new InvalidArgumentException(
                'Cannot proceed asset-update! Asset-Id not set in message.'
            );
        }

        $assetsEntity = $this->assetRepository->findOneBy(['resourceId' => $resourceId]);
        if (null !== $assetsEntity) {
            $this->assetManager->remove($assetsEntity);
            $this->flush($this->assetManager);
        }
    }

    /**
     * @return AssetStorage
     * @throws \Exception
     */
    private function getNewAssetsStorage()
    {
        return new AssetStorage();
    }

    /**
     * @return AssetStorageMetaData
     */
    private function getNewAssetsStorageMetaData()
    {
        return new AssetStorageMetaData();
    }
}
