<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Crefopay\Processor;

use Doctrine\ORM\EntityManager as NotificationManager;
use Lifestyle\Sylius\Crefopay\Entity\CrefopayNotification;
use Payum\Core\Bridge\Spl\ArrayObject;
use Psr\Log\LoggerInterface;
use Sylius\Component\Resource\Factory\Factory as NotificationResourceFactory;
use Upg\Library\Mns\ProcessorInterface;

/**
 * Class NotificationProcessor
 *
 * @package Lifestyle\Sylius\Crefopay\Processor
 */
class NotificationProcessor implements ProcessorInterface
{
    /**
     * @var NotificationResourceFactory
     */
    protected $notificationFactory;
    /**
     * @var NotificationManager
     */
    protected $notificationManager;
    /**
     * @var CrefopayNotification
     */
    protected $notification;
    /**
     * @var LoggerInterface
     */
    protected $notificationLogger;

    /**
     * NotificationProcessor constructor.
     *
     * @param NotificationResourceFactory $notificationFactory
     * @param NotificationManager         $notificationManager
     * @param LoggerInterface             $notificationLogger
     */
    public function __construct(
        NotificationResourceFactory $notificationFactory,
        NotificationManager $notificationManager,
        LoggerInterface $notificationLogger
    ) {
        $this->notificationFactory = $notificationFactory;
        $this->notificationManager = $notificationManager;
        $this->notificationLogger = $notificationLogger;
    }

    /**
     * @inheritDoc
     */
    public function sendData(
        $merchantID,
        $storeID,
        $orderID,
        $captureID,
        $merchantReference,
        $paymentReference,
        $userID,
        $amount,
        $currency,
        $transactionStatus,
        $orderStatus,
        $additionalData,
        $timestamp,
        $version
    ) {
        $this->notification = $this->notificationFactory->createNew();
        $this->notification
            ->setProcessed(false)
            ->setMerchantId($merchantID? : '')
            ->setStoreId($storeID? : '')
            ->setOrderId(intval($orderID? : 0))
            ->setCaptureId($captureID? : '')
            ->setMerchantReference($merchantReference? : '')
            ->setPaymentReference($paymentReference? : '')
            ->setUserId(intval($userID? : 0))
            ->setAmount(floatval($amount? : 0.00))
            ->setCurrency($currency? : '')
            ->setTransactionStatus($transactionStatus? : '')
            ->setOrderStatus($orderStatus? : '')
            ->setAdditionalData($additionalData? : '')
            ->setTimestamp($timestamp? (new \DateTime())->setTimestamp(intval($timestamp)) : new \DateTime())
            ->setVersion($version? : '')
            ->setErrorMessage('')
        ;
    }

    /**
     * @param ArrayObject  $requestData
     * @param string $errorMessage
     */
    public function setFailure(ArrayObject $requestData, string $errorMessage): void
    {
        $this->sendData(
            $requestData->get('merchantID', ''),
            $requestData->get('storeID', ''),
            $requestData->get('orderID', 0),
            $requestData->get('captureID', ''),
            $requestData->get('merchantReference', ''),
            $requestData->get('paymentReference', ''),
            $requestData->get('userID', 0),
            $requestData->get('amount', ''),
            $requestData->get('currency', ''),
            $requestData->get('transactionStatus', ''),
            $requestData->get('orderStatus', ''),
            $requestData->get('additionalData', ''),
            $requestData->get('timestamp', null),
            $requestData->get('version', '')
        );
        $this->notification->setErrorMessage($errorMessage);
    }

    /**
     * @param string $tokenHash
     */
    public function setTokenHash(string $tokenHash): void
    {
        $this->notification->setTokenHash($tokenHash);
    }

    /**
     * @inheritDoc
     */
    public function run()
    {
        try {
            $this->notificationManager->persist($this->notification);
            $this->notificationManager->flush();
        } catch (\Exception $exception) {
            $this->log($exception->getMessage(), 'notificationEntity', $this->notification);
        }
    }

    /**
     * @param string                     $message
     * @param string                     $type
     * @param array|CrefopayNotification $data
     */
    public function log(string $message, string $type, $data): void
    {
        $this->notificationLogger->critical($message, [$type => print_r($data, true)]);
    }
}
