<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Crefopay\Action\HostedPageBefore\Api;

use Lifestyle\Sylius\Crefopay\Request\HostedPageBefore\Api\CaptureLateTransaction;
use Lifestyle\Sylius\Crefopay\Request\HostedPageBefore\Api\CaptureTransaction;
use Payum\Core\Action\ActionInterface;
use Payum\Core\ApiAwareInterface;
use Payum\Core\Bridge\Spl\ArrayObject;
use Payum\Core\Exception\RequestNotSupportedException;
use Payum\Core\GatewayAwareInterface;
use Sylius\Component\Core\Model\OrderInterface;
use Upg\Library\PaymentMethods\Methods;

/**
 * Class CaptureTransactionAction
 *
 * @package Lifestyle\Sylius\Crefopay\Action\HostedPageBefore\Api
 */
class CaptureTransactionAction extends BaseApiAwareAction implements ActionInterface, GatewayAwareInterface, ApiAwareInterface
{
    /**
     * @var array
     */
    protected $supportedPayments = [
        Methods::PAYMENT_METHOD_TYPE_PAYPAL,
        Methods::PAYMENT_METHOD_TYPE_SU,
        Methods::PAYMENT_METHOD_TYPE_CC,
        Methods::PAYMENT_METHOD_TYPE_CC3D,
    ];

    /**
     * @param CaptureTransaction|CaptureLateTransaction $request
     */
    public function execute($request)
    {
        RequestNotSupportedException::assertSupports($this, $request);
        $details = ArrayObject::ensureArrayObject($request->getModel());

        // only run capture if not captured yet
        if (false === $details->get('captured', false)
            && 'MERCHANTPENDING' === $details->get('transactionStatus', 'NEW')
        ) {
            try {
                // do not capture for some payments (do not capture RECHNUNG|LASTSCHRIFT|VORKASSE they need special handling)
                if (true === $details->get('reserved', false)
                    && in_array($details['paymentMethod'], $this->supportedPayments)
                ) {
                    /** @var OrderInterface $order */
                    $order = $request->getFirstModel()->getOrder();

                    $captureTransactionRequest = $this->api->getRequestModelFactory()
                        ->capture()
                        ->setOrderID($order->getId())
                        ->setCaptureID($order->getNumber())
                        ->setAmount(
                            $this->api->getRequestModelFactory()->amount()
                                ->setAmount($order->getTotal())
                        )
                    ;

                    $this->api->addMacToRequest($captureTransactionRequest, $order->getId());

                    $response = $this->api->getApiFactory()->capture($captureTransactionRequest)->sendRequest();
                    if (0 === $response->getData('resultCode')) {
                        $details['captured'] = true;
                        $details['paymentStatus'] = $response->getData('status');
                    }
                }
            } catch (\Exception $exception) {
                $detailMessages = $details->get('messages', []);
                $detailMessages[__CLASS__ . '::' . __FUNCTION__] = $exception->getMessage();
                $details->defaults([
                    'messages' => $detailMessages
                ]);
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function supports($request)
    {
        return
            $request instanceof CaptureTransaction &&
            $request->getModel() instanceof \ArrayAccess
        ;
    }
}
