<?php
/**
 * Lifestyle Webconsulting GmbH
 *
 * LICENSE: This Software is the property of Lifestyle Webconsulting GmbH (Aschaffenburg, Germany)
 * and is protected by copyright law - it is NOT Freeware.
 *
 * Any unauthorized use of this software without a valid license
 * is a violation of the license agreement and will be prosecuted by
 * civil and criminal law.
 *
 * @author      r.stetter
 * @copyright   2019 Lifestyle Webconsulting GmbH
 * @link        http://www.life-style.de
 */

namespace Lifestyle\Sylius\Crefopay\Action\HostedPageBefore;

use Lifestyle\Sylius\Crefopay\Request\HostedPageBefore\Api\CreateTransaction;
use Lifestyle\Sylius\Crefopay\Request\HostedPageBefore\Api\ExternalPaymentSuccess;
use Lifestyle\Sylius\Crefopay\Request\HostedPageBefore\Api\ResetTransaction;
use Payum\Core\Action\ActionInterface;
use Payum\Core\Bridge\Spl\ArrayObject;
use Payum\Core\GatewayAwareInterface;
use Payum\Core\GatewayAwareTrait;
use Payum\Core\Reply\HttpRedirect;
use Payum\Core\Reply\HttpResponse;
use Payum\Core\Request\Authorize;
use Payum\Core\Exception\RequestNotSupportedException;
use Payum\Core\Request\RenderTemplate;

/**
 * Class AuthorizeAction
 *
 * @package Lifestyle\Sylius\Crefopay\Action\HostedPageBefore
 */
class AuthorizeAction implements ActionInterface, GatewayAwareInterface
{
    use GatewayAwareTrait;

    /**
     * @var string
     */
    protected $templateName;

    /**
     * AuthorizeAction constructor.
     *
     * @param string $templateName
     */
    public function __construct(string $templateName)
    {
        $this->templateName = $templateName;
    }

    /**
     * {@inheritDoc}
     *
     * @param Authorize $request
     */
    public function execute($request)
    {
        RequestNotSupportedException::assertSupports($this, $request);

        $model = ArrayObject::ensureArrayObject($request->getModel());
        $model['state'] = 'none';

        // save the url parameters
        if (count($request->getToken()->getUrlParameters()) > 0) {
            foreach ($request->getToken()->getUrlParameters() as $urlParameterKey => $urlParameterValue) {
                $model[$urlParameterKey] = $urlParameterValue;
            }
        }

        $render = false;
        switch ($model['state']) {
            case 'confirm':
                if ($model->validatedKeysSet(['paymentMethod'])
                    && $model->validateNotEmpty(['paymentMethod'])
                ) {
                    // redirect to confirm page
                    throw new HttpRedirect($request->getToken()->getAfterUrl());
                }
                break;
            case 'success':
                $externalPaymentSuccessRequest = new ExternalPaymentSuccess($request->getToken());
                $externalPaymentSuccessRequest->setModel($request->getFirstModel());
                $externalPaymentSuccessRequest->setModel($request->getModel());

                $this->gateway->execute($externalPaymentSuccessRequest);
                break;
            case 'failure':
                // external payment (eg. paypal|sofort) error handling
                // reset transaction and render hosted page again
                $resetTransactionRequest = new ResetTransaction($request->getToken());
                $resetTransactionRequest->setModel($request->getFirstModel());
                $resetTransactionRequest->setModel($request->getModel());

                $this->gateway->execute($resetTransactionRequest);

                $render = true;
                break;
            default:
                $createTransactionRequest = new CreateTransaction($request->getToken());
                $createTransactionRequest->setModel($request->getFirstModel());
                $createTransactionRequest->setModel($request->getModel());

                $this->gateway->execute($createTransactionRequest);

                $render = true;
                break;
        }

        if (true === $render) {
            $renderTemplate = new RenderTemplate($this->templateName, [
                'order' => $request->getFirstModel()->getOrder(),
                'paymentFrameUrl' => $model['paymentFrameUrl'],
            ]);
            $this->gateway->execute($renderTemplate);

            throw new HttpResponse($renderTemplate->getResult());
        }
    }

    /**
     * {@inheritDoc}
     */
    public function supports($request)
    {
        return
            $request instanceof Authorize &&
            $request->getModel() instanceof \ArrayAccess
        ;
    }
}
